#!/bin/bash

##################################################################################################################
#                                                                                                                #
#                                daccord Server v.2.2.x - Installation script                                    #
#                                                                                                                #
##################################################################################################################

# Set stage to either "DEVELOPMENT" or "PRODUCTION" to influence what repository the rpm files are pulled from
STAGE="PRODUCTION"
# Timestamp variable used to timestamp log entries
NOW=$(date +"%d-%m-%Y %T")
# Date variable used to give the logfile a name including the current date + hour & minute of creation
DATE=$(date +"%Y-%m-%d-%H-%M-%S")
# Logfile variable that holds the name of the logifle including the $DATE variable from above
LOGFILE=daccord-basis-install-${DATE}.log

# If the user id isn't 0, the script isn't being run as root and gives out a warning that it needs to be run as root to run successfully
CHECK_ROOTPRIVILEGE(){
    if [[ $EUID -ne 0 ]]; then
        echo -e "\033[0;31mThis script must be run with root privileges! Please verify that you have the needed privileges and restart the installation!\033[m"
        EXIT
    fi
}

# Check if the script is running on openSUSE or SLES or Red Hat
CHECK_OS(){
    cat /etc/os-release >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        OS=$(cat /etc/os-release | grep -e "^NAME=")
        if [[ ${OS} == 'NAME="openSUSE Leap"' ]] || [[ ${OS} == 'NAME="SLES"' ]]; then
            REDHAT="FALSE"
        else
            REDHAT="TRUE"
        fi
    else
        REDHAT="TRUE"
    fi
}

# Check if the stage variable is set to "DEVELOPMENT" and warn about it if set as such
CHECK_DEV(){
    if [[ $STAGE == "DEVELOPMENT" ]]; then
        echo -e "\033[0;31m\nWARNING! - This script is running in DEVELOPMENT mode.\033[m"
        echo -e "\033[0;31mIt will install or update from DEVELOPMENT repositories.\033[m"
    fi
}

# Creates a file to let the script notice on the next run if the previous installation didn't run through entirely
SET_PREVINSTALL_FLAG(){
    mkdir -p /opt/GuH/ >> ./${LOGFILE} 2>&1
    touch /opt/GuH/daccord_basis_install_flag >> ./${LOGFILE} 2>&1
}

# Check if the file that is being created by the above funtion exists on this system
CHECK_PREVINSTALL_FLAG(){
    if [[ -e /opt/GuH/daccord_basis_install_flag ]]; then
        PREVINSTALL=1
    fi
}

CANCEL(){
    echo "Canceling the install script!"
    echo "[${NOW} install.sh] Script has been canceled." >> ./${LOGFILE}
    exit 1
}

EXIT(){
    exit 1
}

PRESS_ANY_KEY(){
    echo -e "Press any key to continue."
    read -sn 1
}

# Check if the system currently already has the daccord server software installed
CHECK_INST_CURRENT(){
    DACCORD_CURRENT_STAT=-1
    if [[ -e /opt/GuH/ ]]; then
        DACCORD_CURRENT_STAT=1
    fi
}

CHECK_SERVER_TYPE(){
    SERVERTYPE=""
    if [[ -e /opt/GuH/daccord-system-gadget ]]; then
        SERVERTYPE="p"
    else
        SERVERTYPE="s"
    fi
}

SERVER_TYPE(){
    SERVERTYPEQUERY=
    while ! [[ "${SERVERTYPEQUERY}" =~ ^([yYnN])$ ]]; do
        echo -e "\n\033[0;36mThe daccord system needs at least one primary daccord server which will have the required core components installed.\033[m"
        read -p $'\033[0;36mDo you want this server to be configured as your primary daccord server? [y/n] \033[m' SERVERTYPEQUERY
        if [[ "${SERVERTYPEQUERY}" =~ ^([yY])$ ]]; then
            export SERVERTYPE="p"
        elif [[ "${SERVERTYPEQUERY}" =~ ^([nN])$ ]]; then
            export SERVERTYPE="s"
        fi
    done
}

SELFSIGNEDCERT(){
cat <<EOF> /opt/GuH/RuntimeEngine/conf/req.conf
[ req ]
default_bits              = 4096  # RSA key size
encrypt_key               = no  # Do not protect private key
default_md                = sha256  # MD to use
utf8                      = yes  # Input is UTF-8
string_mask               = utf8only  # Emit UTF-8 strings
prompt                    = no  # Do not prompt for DN
distinguished_name        = server_dn  # DN template
req_extensions            = server_reqext  # Desired extensions

[ server_dn ]
commonName                = @DNS_OR_IP@  # Should match a SAN under alt_names

[ server_reqext ]
basicConstraints          = CA:FALSE
keyUsage                  = critical,digitalSignature,keyEncipherment
extendedKeyUsage          = serverAuth
subjectKeyIdentifier      = hash
subjectAltName            = @alt_names

[alt_names]
@ENTRY@
EOF
cat <<EOF> /opt/GuH/RuntimeEngine/conf/v3.ext
subjectAltName         = @ENTRY@
EOF
    sed -i "s/@DNS_OR_IP@/${REDNSORIP}/g" /opt/GuH/RuntimeEngine/conf/req.conf
    if [[ ${DNS} == "TRUE" ]]; then
        sed -i "s/@ENTRY@/DNS.1                     = ${REDNSORIP}/g" /opt/GuH/RuntimeEngine/conf/req.conf
        sed -i "s/@ENTRY@/DNS:${REDNSORIP}/g" /opt/GuH/RuntimeEngine/conf/v3.ext
    else
        sed -i "s/@ENTRY@/IP.1                      = ${REDNSORIP}/g" /opt/GuH/RuntimeEngine/conf/req.conf
        sed -i "s/@ENTRY@/IP:${REDNSORIP}/g" /opt/GuH/RuntimeEngine/conf/v3.ext
    fi
    openssl req -new -keyout /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.key -out /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.req -batch -config /opt/GuH/RuntimeEngine/conf/req.conf >> ./${LOGFILE} 2>&1
    if [[ ${DNS} == "TRUE" ]]; then
        openssl x509 -req -in /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.req -CA /opt/GuH/RuntimeEngine/conf/PlaceholderCA.crt -CAkey /opt/GuH/RuntimeEngine/conf/PlaceholderCA.key -out /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.crt -days 3650 -sha256 -CAcreateserial -extfile /opt/GuH/RuntimeEngine/conf/v3.ext >> ./${LOGFILE} 2>&1
    else
        openssl x509 -req -in /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.req -CA /opt/GuH/RuntimeEngine/conf/PlaceholderCA.crt -CAkey /opt/GuH/RuntimeEngine/conf/PlaceholderCA.key -out /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.crt -days 3650 -sha256 -CAcreateserial -extfile /opt/GuH/RuntimeEngine/conf/v3.ext >> ./${LOGFILE} 2>&1
    fi
    cat /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.crt /opt/GuH/RuntimeEngine/conf/PlaceholderCA.crt > /opt/GuH/RuntimeEngine/conf/chain.txt
    openssl pkcs12 --password pass:changeit -export -inkey /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.key -in /opt/GuH/RuntimeEngine/conf/chain.txt -out /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.pkcs12 -name "${REDNSORIP}" >> ./${LOGFILE} 2>&1
    sed -i 's\keystoreFile="/opt/GuH/RuntimeEngine/conf/placeholder.pkcs12"\keystoreFile="/opt/GuH/RuntimeEngine/conf/'"${REDNSORIP}"'.pkcs12"\g' /opt/GuH/RuntimeEngine/conf/server.xml >> ./${LOGFILE} 2>&1
    rm /opt/GuH/RuntimeEngine/conf/chain.txt /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.key /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.crt /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.req /opt/GuH/RuntimeEngine/conf/req.conf /opt/GuH/RuntimeEngine/conf/v3.ext >> /dev/null 2>&1
    chown root:RuntimeEngine /opt/GuH/RuntimeEngine/conf/* >> ./${LOGFILE} 2>&1
    chmod ug+rwx /opt/GuH/RuntimeEngine/conf/* >> ./${LOGFILE} 2>&1
}

OUTPUT_WELCOME(){
    clear
    echo -e "\033[0;36m################################################################################\033[m"
    echo -e "\033[0;36m#                                                                              #\033[m"
    echo -e "\033[0;36m#\033[m                \033[mdaccord Server v.2.2.x - Installation script\033[m                  \033[0;36m#\033[m"
    echo -e "\033[0;36m#                                                                              #\033[m"
    echo -e "\033[0;36m################################################################################\033[m"
    echo -e " "
    echo -e "For detailed information please refer to the logfile: ${PWD}/${LOGFILE}"
}

ENV_VARIABLES(){
    REURLSCHEMA=
    while ! [[ "${REURLSCHEMA}" == [iIsS] ]]; do
        echo -e "\n\033[0;36mThe different daccord components communicate with each other. It is possibile to configure if the communication should be done securely (HTTPS and port 55443) or insecurely (HTTP and port 55080).\033[m"
        read -p $'\033[0;36mPlease choose either [i] for "Insecure" or [s] for "Secure" communication! [i/s] \033[m' REURLSCHEMA
        if [[ "${REURLSCHEMA}" == [iI] ]]; then
            export REURLSCHEMACHOSEN="http://"
            export REURLPORT=":55080"
        elif [[ "${REURLSCHEMA}" == [sS] ]]; then
            export REURLSCHEMACHOSEN="https://"
            export REURLPORT=":55443"
        fi
    done
    REDNSORIP=
    while ! [[ "${REDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z|-]*[0-9|a-zA-Z]$ ]]; do
        echo -e "\n\033[0;36mThis server needs to be accessible by the other components over the network.\033[m"
        read -p $'\033[0;36mPlease enter either the valid IP address or resolvable DNS name of this server: \033[m' REDNSORIP
        if [[ "${REDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z|-]*[0-9|a-zA-Z]$ ]]; then
            if [[ "${REDNSORIP}" =~ ^[0-9]+[.|0-9]*[0-9]$ ]]; then
                ip a | grep ${REDNSORIP} >> /dev/null 2>&1
                if [[ ! $? == 0 ]]; then
                    echo -e "\nThe IP Address that has been provided is not bound on any network adapter device on this server!"
                    REDNSORIP=
                else
                    DNS="FALSE"
                    export REDNSORIP
                fi
            else
                rpm -qa | grep -i '^bind-utils-[0-9]' >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    echo "[${NOW} install.sh] The 'bind-utils' package is already installed, proceeding with URL check." >> ./${LOGFILE}
                else
                    echo "[${NOW} install.sh] The 'bind-utils' package is not yet installed, proceeding with 'bind-utils' installation." >> ./${LOGFILE}
                    if [[ ${REDHAT} == "TRUE" ]]; then
                        yum install -y bind-utils >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh bind-utils*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'bind-utils' package." >> ./${LOGFILE}
                            fi
                        fi
                    else
                        zypper --non-interactive in bind-utils >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh bind-utils*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'bind-utils' package." >> ./${LOGFILE}
                            fi
                        fi
                    fi
                fi
                nslookup ${REDNSORIP} | grep -i "can't find" >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    echo -e "\nThe DNS Name that has been provided is not resolvable!"
                    REDNSORIP=
                else
                    DNS="TRUE"
                    export REDNSORIP
                fi
            fi
        fi
    done
    export DAC_RUNTIMEENGINE_URL="${REURLSCHEMACHOSEN}${REDNSORIP}${REURLPORT}/"
    if [[ ${SERVERTYPE} = [sS] ]]; then
        SYSURLSCHEMA=
        while ! [[ "${SYSURLSCHEMA}" == [iIsS] ]]; do
            echo -e "\n\033[0;36mThe different daccord components communicate with each other either securely (HTTPS and port 55443) or insecurely (HTTP and port 55080) depending on how the primary daccord server has been configured.\033[m"
            read -p $'\033[0;36mPlease choose either [i] for "Insecure" or [s] for "Secure" communication to the primary daccord server! [i/s] \033[m' SYSURLSCHEMA
            if [[ "${SYSURLSCHEMA}" == [iI] ]]; then
                export SYSURLSCHEMACHOSEN="http://"
                export SYSURLPORT=":55080"
            elif [[ "${SYSURLSCHEMA}" == [sS] ]]; then
                export SYSURLSCHEMACHOSEN="https://"
                export SYSURLPORT=":55443"
            fi
        done
        SYSDNSORIP=
        while ! [[ "${SYSDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z]*[0-9|a-zA-Z]$ ]]; do
            echo -e "\n\033[0;36mThis server needs to access the primary daccord server over the network.\033[m"
            read -p $'\033[0;36mPlease enter either the valid IP address or resolvable DNS name of the primary daccord server: \033[m' SYSDNSORIP
            if [[ "${SYSDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z]*[0-9|a-zA-Z]$ ]]; then
                export SYSDNSORIP
                export DAC_SYSTEM_URL="${SYSURLSCHEMACHOSEN}${SYSDNSORIP}${SYSURLPORT}/system/"
                rpm -qa | grep -i '^wget-[0-9]' >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    echo "[${NOW} install.sh] The 'wget' package is already installed, proceeding with URL check." >> ./${LOGFILE}
                else
                    echo "[${NOW} install.sh] The 'wget' package is not yet installed, proceeding with 'wget' installation." >> ./${LOGFILE}
                    if [[ ${REDHAT} == "TRUE" ]]; then
                        yum install -y wget >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh wget*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'wget' package." >> ./${LOGFILE}
                            fi
                        fi
                    else
                        zypper --non-interactive in wget >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh wget*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'wget' package." >> ./${LOGFILE}
                            fi
                        fi
                    fi
                    while true; do
                        rpm -qa | grep -i '^wget-[0-9]' >> /dev/null 2>&1
                        if [[ $? == 0 ]]; then
                            sleep 2
                            break
                        fi
                        echo -n '.'
                        sleep 1
                    done
                fi
                wget --no-check-certificate --timeout=5 --tries=1 ${DAC_SYSTEM_URL} >> ./${LOGFILE} 2>&1
                rm ./index.html >> /dev/null 2>&1
                if [[ ! $? == 0 ]]; then
                    echo -e "\nThe primary server was not reachable with the provided configuration. Please check and enter the configuration again!"
                    echo -e "Provided configuration: ${DAC_SYSTEM_URL}"
                    echo ""
                    DAC_SYSTEM_URL=""
                    SYSDNSORIP=""
                    SYSURLSCHEMA=""
                    while ! [[ "${SYSURLSCHEMA}" == [iIsS] ]]; do
                        echo -e "\n\033[0;36mThe different daccord components communicate with each other either securely (HTTPS and port 55443) or insecurely (HTTP and port 55080) depending on how the primary daccord server has been configured.\033[m"
                        read -p $'\033[0;36mPlease choose either [i] for "Insecure" or [s] for "Secure" communication to the primary daccord server! [i/s] \033[m' SYSURLSCHEMA
                        if [[ "${SYSURLSCHEMA}" == [iI] ]]; then
                            export SYSURLSCHEMACHOSEN="http://"
                            export SYSURLPORT=":55080"
                        elif [[ "${SYSURLSCHEMA}" == [sS] ]]; then
                            export SYSURLSCHEMACHOSEN="https://"
                            export SYSURLPORT=":55443"
                        fi
                    done
                fi
            fi
        done
        export DAC_SYSTEM_URL="${SYSURLSCHEMACHOSEN}${SYSDNSORIP}${SYSURLPORT}/system/"
    else
        export DAC_SYSTEM_URL="${DAC_RUNTIMEENGINE_URL}system/"
    fi
    echo "DAC_RUNTIMEENGINE_URL='${DAC_RUNTIMEENGINE_URL}'" >> /etc/environment
    echo "DAC_SYSTEM_URL='${DAC_SYSTEM_URL}'" >> /etc/environment
    source /etc/environment >> ./${LOGFILE} 2>&1
}

NEO4JINITIALPASS(){
    NEO4JINITIALPASS=
    while ! [[ "${NEO4JINITIALPASS}" =~ ^[0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]+[.|0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]*[0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]{7,}$ ]]; do
        read -r -s -p $'\n\033[0;36mPlease provide a password with a minimum of 8 alphanumeric and these {}()´@*~.,!?_-+# allowed special characters for the "neo4j" database user: \033[m' NEO4JINITIALPASS
    done
    echo " "
    NEO4JINITIALPASS2=
    while ! [[ "${NEO4JINITIALPASS2}" =~ ^[0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]+[.|0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]*[0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]{7,}$ ]]; do
        read -r -s -p $'\n\033[0;36mPlease re-enter the password to confirm it: \033[m' NEO4JINITIALPASS2
    done
    if [[ "${NEO4JINITIALPASS}" != "${NEO4JINITIALPASS2}" ]]; then
        echo " "
        echo -e "\n\033[0;31mThe provided passwords do not match, please try again!\033[m"
        NEO4JINITIALPASS
    fi
}

NEO4JUSER(){
    echo " "
    echo " "
    NEO4JUSER=
    while [[ -z "${NEO4JUSER}" ]]; do
        read -p $'\033[0;36mPlease provide an alphanumeric username for the system database access user (also used by background components). Example: daccord. Please remember this username for future configuration: \033[m' NEO4JUSER
        if [[ -z "${NEO4JUSER}" ]]; then
            echo -e "\n\033[0;35mPlease provide an alphanumeric username for the system database access user (also used by background components). Example: daccord. Please remember this username for future configuration: \033[m"
        fi
    done
    echo -e "\n\033[0;36mThe following username has been provided:\033[m ${NEO4JUSER}"
}

NEO4JPASS(){
    NEO4JPASS=
    while ! [[ "${NEO4JPASS}" =~ ^[0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]+[.|0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]*[0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]{7,}$ ]]; do
        read -r -s -p $'\n\033[0;36mPlease provide a password with a minimum of 8 alphanumeric and these {}()´@*~.,!?_-+# allowed special characters for your "'${NEO4JUSER}'" database user: ' NEO4JPASS
    done
    echo " "
    NEO4JPASS2=
    while ! [[ "${NEO4JPASS2}" =~ ^[0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]+[.|0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]*[0-9|a-zA-Z|\{\}\(\)\´\@\*\~\.\,\!\?\-\_\+\#]{7,}$ ]]; do
        read -r -s -p $'\n\033[0;36mPlease re-enter the password to confirm it: \033[m' NEO4JPASS2
    done
    if [[ "${NEO4JPASS}" != "${NEO4JPASS2}" ]]; then
        echo " "
        echo -e "\n\033[0;31mThe provided passwords do not match, please try again!\033[m"
        NEO4JPASS
    fi
}

# Give the user the option to install neo4j on a SUSE system through zypper (online modus) if not already installed
CHECK_NEO4J_SUSE(){
    rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -e "\nThe System Database is already installed on this system. Please have your login credentials on hand for the configuration at a later stage."
        echo "[${NOW} install.sh] neo4j is already installed on this system." >> ./${LOGFILE}
    else
        EINGABE=
        while [[ ${EINGABE} != [lLeE] ]]; do
            echo -e "\n\033[0;36mThe daccord software requires a database to store data.\033[m"
            read -p $'\033[0;36mDo you want to install the System Database locally [l] or use an external [e] instance? [l/e] \033[m' EINGABE
            if [[ ${EINGABE} != [lLeE] ]]; then
                echo -e "\n\033[0;35mPlease choose between [l] to install the System Database locally or [e] to use an external instance!\033[m"
            fi
        done
        if [[ ${EINGABE} == [lL] ]]; then
            echo "[${NOW} install.sh] User has chosen to install neo4j locally." >> ./${LOGFILE}
            NEO4JINITIALPASS
            NEO4JUSER
            NEO4JPASS
            echo " "
            echo " "
            echo -n "Please wait while the System Database is being installed..."
            echo "[${NOW} install.sh] Checking if the required 'insserv-compat' package is already installed." >> ./${LOGFILE}
            rpm -qa | grep -i '^insserv-compat-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                echo "[${NOW} install.sh] The 'insserv-compat' package is already installed, proceeding with neo4j installation." >> ./${LOGFILE}
            else
                echo "[${NOW} install.sh] The 'insserv-compat' package is not yet installed, proceeding with 'insserv-compat' installation." >> ./${LOGFILE}
                zypper --non-interactive in insserv-compat >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    while true; do
                        rpm -qa | grep -i '^insserv-compat-[0-9]' >> /dev/null 2>&1
                        if [[ $? == 0 ]]; then
                            sleep 3
                            break
                        fi
                        echo -n '.'
                        sleep 3
                    done
                else
                    rpm -Uvh insserv-compat* >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        while true; do
                            rpm -qa | grep -i '^insserv-compat-[0-9]' >> /dev/null 2>&1
                            if [[ $? == 0 ]]; then
                                sleep 3
                                break
                            fi
                            echo -n '.'
                            sleep 3
                        done
                    else
                        echo "[${NOW} install.sh] Unable to install the 'insserv-compat' package" >> ./${LOGFILE}
                    fi
                fi
            fi
            zypper ar --refresh http://yum.neo4j.com/stable/5 neo4j >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh neo4j >> ./${LOGFILE} 2>&1
            zypper --non-interactive in neo4j-5.25.1-1 >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                while true; do
                    rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
                    if [[ $? == 0 ]]; then
                        sleep 3
                        break
                    fi
                    echo -n '.'
                    sleep 3
                done
            else
                rpm -Uvh neo4j* >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    while true; do
                        rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
                        if [[ $? == 0 ]]; then
                            sleep 3
                            break
                        fi
                        echo -n '.'
                        sleep 3
                    done
                else
                    echo "[${NOW} install.sh] An error occured while trying to install neo4j." >> ./${LOGFILE}
                    echo -n "an error occured while trying to install the System Database, please refer to logfile: ${PWD}/${LOGFILE}"
                    CANCEL
                fi
            fi
            zypper rr neo4j >> ./${LOGFILE} 2>&1
            if [[ $? != 0 ]]; then
                while true; do
                    zypper lr neo4j >> /dev/null 2>&1
                    if [[ $? == 0 ]]; then
                        zypper rr neo4j >> ./${LOGFILE} 2>&1
                    else
                        break
                    fi
                    sleep 3
                done
            fi

            wget -T 10 https://github.com/neo4j/apoc/releases/download/5.25.1/apoc-5.25.1-core.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo "[${NOW} install.sh] The apoc-5.25.1-core.jar file has been successfully downloaded." >> ./${LOGFILE}
            else
                sleep 5 >> /dev/null 2>&1
                wget -T 10 https://github.com/neo4j/apoc/releases/download/5.25.1/apoc-5.25.1-core.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo "[${NOW} install.sh] The apoc-5.25.1-core.jar file has been successfully downloaded." >> ./${LOGFILE}
                else
                    wget -T 10 http://github.com/neo4j/apoc/releases/download/5.25.1/apoc-5.25.1-core.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        echo "[${NOW} install.sh] The apoc-5.25.1-core.jar file has been successfully downloaded." >> ./${LOGFILE}
                    else
                        sleep 5 >> /dev/null 2>&1
                        wget -T 10 http://github.com/neo4j/apoc/releases/download/5.25.1/apoc-5.25.1-core.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                        if [[ $? == 0 ]]; then
                            echo "[${NOW} install.sh] The apoc-5.25.1-core.jar file has been successfully downloaded." >> ./${LOGFILE}
                        else
                            echo "[${NOW} install.sh] An error occured while trying to download the apoc-5.25.1-core.jar file." >> ./${LOGFILE}
                            echo -e "\033[0;31mAn error occured while trying to download the apoc-5.25.1-core.jar file, please refer to logfile:\033[m ${PWD}/${LOGFILE}"
                            PRESS_ANY_KEY
                        fi
                    fi
                fi
            fi

            wget -T 10 https://www.daccord.de/software/daccord/resources/neo4j-daccord-event-extension-2.2.0.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo "[${NOW} install.sh] The neo4j-daccord-event-extension-2.2.0.jar file has been successfully downloaded." >> ./${LOGFILE}
            else
                sleep 5 >> /dev/null 2>&1
                wget -T 10 https://www.daccord.de/software/daccord/resources/neo4j-daccord-event-extension-2.2.0.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo "[${NOW} install.sh] The neo4j-daccord-event-extension-2.2.0.jar file has been successfully downloaded." >> ./${LOGFILE}
                else
                    wget -T 10 http://www.daccord.de/software/daccord/resources/neo4j-daccord-event-extension-2.2.0.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        echo "[${NOW} install.sh] The neo4j-daccord-event-extension-2.2.0.jar file has been successfully downloaded." >> ./${LOGFILE}
                    else
                        sleep 5 >> /dev/null 2>&1
                        wget -T 10 http://www.daccord.de/software/daccord/resources/neo4j-daccord-event-extension-2.2.0.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                        if [[ $? == 0 ]]; then
                            echo "[${NOW} install.sh] The neo4j-daccord-event-extension-2.2.0.jar file has been successfully downloaded." >> ./${LOGFILE}
                        else
                            echo "[${NOW} install.sh] An error occured while trying to download the neo4j-daccord-event-extension-2.2.0.jar file." >> ./${LOGFILE}
                            echo -e "\033[0;31mAn error occured while trying to download the neo4j-daccord-event-extension-2.2.0.jar file, please refer to logfile:\033[m ${PWD}/${LOGFILE}"
                            PRESS_ANY_KEY
                        fi
                    fi
                fi
            fi

            zypper se -i openjdk | grep 21 >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                CHECK_JAVA_VERSION=$(java -version 2>&1 | grep -o --max-count=1 '"21.')
                if [[ ${CHECK_JAVA_VERSION} != '"21.' ]]; then
                    zypper se -i openjdk | grep 17 >> /dev/null 2>&1
                    if [[ $? == 0 ]]; then
                        CHECK_JAVA_VERSION=$(java -version 2>&1 | grep -o --max-count=1 '"17.')
                        if [[ ${CHECK_JAVA_VERSION} != '"17.' ]]; then
                            update-alternatives --set java /usr/lib64/jvm/jre-21-openjdk/bin/java >> ./${LOGFILE} 2>&1
                        fi
                    else
                        update-alternatives --set java /usr/lib64/jvm/jre-21-openjdk/bin/java >> ./${LOGFILE} 2>&1
                    fi
                fi
            else
                zypper se -i openjdk | grep 17 >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    CHECK_JAVA_VERSION=$(java -version 2>&1 | grep -o --max-count=1 '"17.')
                    if [[ ${CHECK_JAVA_VERSION} != '"17.' ]]; then
                        update-alternatives --set java /usr/lib64/jvm/jre-17-openjdk/bin/java >> ./${LOGFILE} 2>&1
                    fi
                else
                    echo "[${NOW} install.sh] No compatible Java Version found!" >> ./${LOGFILE}
                    echo -e "\033[0;31mNo compatible Java Version found!\033[m"
                    CANCEL
                fi
            fi

            sed -i "s/#server.default_listen_address/server.default_listen_address/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/#initial.dbms.default_database=neo4j/initial.dbms.default_database=daccord.db/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/dbms.connector.https.enabled=true/dbms.connector.https.enabled=false/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/#dbms.security.procedures.unrestricted=my.extensions.example,my.procedures.*/dbms.security.procedures.unrestricted=algo.*,apoc.*/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/#dbms.usage_report.enabled/dbms.usage_report.enabled/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/# server.jvm.additional=-XX:+ExitOnOutOfMemoryError/server.jvm.additional=-XX:+ExitOnOutOfMemoryError/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1

            echo "apoc.export.file.enabled=true" > /etc/neo4j/apoc.conf
            echo "apoc.import.file.enabled=true" >> /etc/neo4j/apoc.conf

            neo4j-admin dbms set-initial-password ${NEO4JINITIALPASS} >> ./${LOGFILE} 2>&1

            rm -f /etc/init.d/neo4j >> ./${LOGFILE} 2>&1
            systemctl enable neo4j.service >> ./${LOGFILE} 2>&1
            systemctl start neo4j.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo "[${NOW} install.sh] neo4j has been successfully installed." >> ./${LOGFILE}
                echo -ne "System Database installed successfully\n"
                echo -n "Please wait while the System Database is being configured. This may take several minutes..."
                until cypher-shell --username neo4j --password ${NEO4JINITIALPASS} --database system 'SHOW DATABASE system;' >> ./${LOGFILE} 2>&1
                do
                    sleep 10
                done
                sleep 3

                cypher-shell --username neo4j --password ${NEO4JINITIALPASS} --database system 'CREATE USER '${NEO4JUSER}' IF NOT EXISTS SET PASSWORD '\"${NEO4JPASS}\"' CHANGE NOT REQUIRED;' >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "System Database configured successfully\n"
                else
                    echo "[${NOW} install.sh] An error occured while trying to prepare neo4j." >> ./${LOGFILE}
                    echo -n "an error occured while trying to prepare the System Database, please refer to logfile: ${PWD}/${LOGFILE}"
                    PRESS_ANY_KEY
                fi
            else
                echo "[${NOW} install.sh] An error occured while trying to start neo4j." >> ./${LOGFILE}
                echo -n "an error occured while trying to start the System Database, please refer to logfile: ${PWD}/${LOGFILE}"
                PRESS_ANY_KEY
            fi
        else
            echo "[${NOW} install.sh] User has chosen not to install neo4j locally." >> ./${LOGFILE}
            echo -e "\nPlease have your login credentials on hand for the configuration at a later stage."
            PRESS_ANY_KEY
        fi
    fi
}

# Give the user the option to install neo4j on a RedHat system through yum (online modus) if not already installed
CHECK_NEO4J_REDHAT(){
    rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -e "\nThe System Database is already installed on this system. Please have your login credentials on hand for the configuration at a later stage."
        echo "[${NOW} install.sh] neo4j is already installed on this system." >> ./${LOGFILE}
    else
        EINGABE=
        while [[ ${EINGABE} != [lLeE] ]]; do
            echo -e "\n\033[0;36mThe daccord software requires a database to store data.\033[m"
            read -p $'\033[0;36mDo you want to install the System Database locally [l] or use an external [e] instance? [l/e] \033[m' EINGABE
            if [[ ${EINGABE} != [lLeE] ]]; then
                echo -e "\n\033[0;35mPlease choose between [l] to install the System Database locally or [e] to use an external instance!\033[m"
            fi
        done
        if [[ ${EINGABE} == [lL] ]]; then
            echo "[${NOW} install.sh] User has chosen to install neo4j locally." >> ./${LOGFILE}
            NEO4JINITIALPASS
            NEO4JUSER
            NEO4JPASS
            echo " "
            echo " "
            echo -n "Please wait while the System Database is being installed..."
            echo "[${NOW} install.sh] Checking if the required 'insserv-compat' package is already installed." >> ./${LOGFILE}
            rpm -qa | grep -i '^insserv-compat-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                echo "[${NOW} install.sh] The 'insserv-compat' package is already installed, proceeding with neo4j installation." >> ./${LOGFILE}
            else
                echo "[${NOW} install.sh] The 'insserv-compat' package is not yet installed, proceeding with 'insserv-compat' installation." >> ./${LOGFILE}
                yum install -y insserv-compat >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    while true; do
                        rpm -qa | grep -i '^insserv-compat-[0-9]' >> /dev/null 2>&1
                        if [[ $? == 0 ]]; then
                            sleep 3
                            break
                        fi
                        echo -n '.'
                        sleep 3
                    done
                else
                    rpm -Uvh insserv-compat* >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        while true; do
                            rpm -qa | grep -i '^insserv-compat-[0-9]' >> /dev/null 2>&1
                            if [[ $? == 0 ]]; then
                                sleep 3
                                break
                            fi
                            echo -n '.'
                            sleep 3
                        done
                    else
                        echo "[${NOW} install.sh] Unable to install the 'insserv-compat' package" >> ./${LOGFILE}
                    fi
                fi
            fi
            rpm --import https://debian.neo4j.com/neotechnology.gpg.key
cat <<EOF>  /etc/yum.repos.d/neo4j.repo
[neo4j]
name=Neo4j RPM Repository
baseurl=https://yum.neo4j.com/stable/5/
enabled=1
gpgcheck=1
EOF
            yum install -y neo4j-5.25.1-1 >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                while true; do
                    rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
                    if [[ $? == 0 ]]; then
                        sleep 3
                        break
                    fi
                    echo -n '.'
                    sleep 3
                done
            else
                rpm -Uvh neo4j* >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    while true; do
                        rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
                        if [[ $? == 0 ]]; then
                            sleep 3
                            break
                        fi
                        echo -n '.'
                        sleep 3
                    done
                else
                    echo "[${NOW} install.sh] An error occured while trying to install neo4j." >> ./${LOGFILE}
                    echo -n "an error occured while trying to install the System Database, please refer to logfile: ${PWD}/${LOGFILE}"
                    CANCEL
                fi
            fi
            rm -fR /etc/yum.repos.d/neo4j.repo >> ./${LOGFILE} 2>&1

            wget -T 10 https://github.com/neo4j/apoc/releases/download/5.25.1/apoc-5.25.1-core.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo "[${NOW} install.sh] The apoc-5.25.1-core.jar file has been successfully downloaded." >> ./${LOGFILE}
            else
                sleep 5 >> /dev/null 2>&1
                wget -T 10 https://github.com/neo4j/apoc/releases/download/5.25.1/apoc-5.25.1-core.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo "[${NOW} install.sh] The apoc-5.25.1-core.jar file has been successfully downloaded." >> ./${LOGFILE}
                else
                    wget -T 10 http://github.com/neo4j/apoc/releases/download/5.25.1/apoc-5.25.1-core.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        echo "[${NOW} install.sh] The apoc-5.25.1-core.jar file has been successfully downloaded." >> ./${LOGFILE}
                    else
                        sleep 5 >> /dev/null 2>&1
                        wget -T 10 http://github.com/neo4j/apoc/releases/download/5.25.1/apoc-5.25.1-core.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                        if [[ $? == 0 ]]; then
                            echo "[${NOW} install.sh] The apoc-5.25.1-core.jar file has been successfully downloaded." >> ./${LOGFILE}
                        else
                            echo "[${NOW} install.sh] An error occured while trying to download the apoc-5.25.1-core.jar file." >> ./${LOGFILE}
                            echo "An error occured while trying to download the apoc-5.25.1-core.jar file, please refer to logfile: ${PWD}/${LOGFILE}"
                            PRESS_ANY_KEY
                        fi
                    fi
                fi
            fi

            wget -T 10 https://www.daccord.de/software/daccord/resources/neo4j-daccord-event-extension-2.2.0.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo "[${NOW} install.sh] The neo4j-daccord-event-extension-2.2.0.jar file has been successfully downloaded." >> ./${LOGFILE}
            else
                sleep 5 >> /dev/null 2>&1
                wget -T 10 https://www.daccord.de/software/daccord/resources/neo4j-daccord-event-extension-2.2.0.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo "[${NOW} install.sh] The neo4j-daccord-event-extension-2.2.0.jar file has been successfully downloaded." >> ./${LOGFILE}
                else
                    wget -T 10 http://www.daccord.de/software/daccord/resources/neo4j-daccord-event-extension-2.2.0.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        echo "[${NOW} install.sh] The neo4j-daccord-event-extension-2.2.0.jar file has been successfully downloaded." >> ./${LOGFILE}
                    else
                        sleep 5 >> /dev/null 2>&1
                        wget -T 10 http://www.daccord.de/software/daccord/resources/neo4j-daccord-event-extension-2.2.0.jar -P /var/lib/neo4j/plugins/ >> ./${LOGFILE} 2>&1
                        if [[ $? == 0 ]]; then
                            echo "[${NOW} install.sh] The neo4j-daccord-event-extension-2.2.0.jar file has been successfully downloaded." >> ./${LOGFILE}
                        else
                            echo "[${NOW} install.sh] An error occured while trying to download the neo4j-daccord-event-extension-2.2.0.jar file." >> ./${LOGFILE}
                            echo "An error occured while trying to download the neo4j-daccord-event-extension-2.2.0.jar file, please refer to logfile: ${PWD}/${LOGFILE}"
                            PRESS_ANY_KEY
                        fi
                    fi
                fi
            fi

            yum list --installed | grep openjdk | grep 21 >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                CHECK_JAVA_VERSION=$(java -version 2>&1 | grep -o --max-count=1 '"21.')
                if [[ ${CHECK_JAVA_VERSION} != '"21.' ]]; then
                    yum list --installed | grep openjdk | grep 17 >> /dev/null 2>&1
                    if [[ $? == 0 ]]; then
                        CHECK_JAVA_VERSION=$(java -version 2>&1 | grep -o --max-count=1 '"17.')
                        if [[ ${CHECK_JAVA_VERSION} != '"17.' ]]; then
                            update-alternatives --set java /usr/lib64/jvm/jre-21-openjdk/bin/java >> ./${LOGFILE} 2>&1
                        fi
                    else
                        update-alternatives --set java /usr/lib64/jvm/jre-21-openjdk/bin/java >> ./${LOGFILE} 2>&1
                    fi
                fi
            else
                yum list --installed | grep openjdk | grep 17 >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    CHECK_JAVA_VERSION=$(java -version 2>&1 | grep -o --max-count=1 '"17.')
                    if [[ ${CHECK_JAVA_VERSION} != '"17.' ]]; then
                        update-alternatives --set java /usr/lib64/jvm/jre-17-openjdk/bin/java >> ./${LOGFILE} 2>&1
                    fi
                else
                    echo "[${NOW} install.sh] No compatible Java Version found!" >> ./${LOGFILE}
                    echo -e "\033[0;31mNo compatible Java Version found!\033[m"
                    CANCEL
                fi
            fi
            sed -i "s/#server.default_listen_address/server.default_listen_address/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/#initial.dbms.default_database=neo4j/initial.dbms.default_database=daccord.db/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/dbms.connector.https.enabled=true/dbms.connector.https.enabled=false/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/#dbms.security.procedures.unrestricted=my.extensions.example,my.procedures.*/dbms.security.procedures.unrestricted=algo.*,apoc.*/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/#dbms.usage_report.enabled/dbms.usage_report.enabled/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/# server.jvm.additional=-XX:+ExitOnOutOfMemoryError/server.jvm.additional=-XX:+ExitOnOutOfMemoryError/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1

            echo "apoc.export.file.enabled=true" > /etc/neo4j/apoc.conf
            echo "apoc.import.file.enabled=true" >> /etc/neo4j/apoc.conf
            echo 'dbms.jvm.additional=-Djava.net.preferIPv4Stack=true' >> /etc/neo4j/neo4j.conf

            neo4j-admin dbms set-initial-password ${NEO4JINITIALPASS} >> ./${LOGFILE} 2>&1

            rm -f /etc/init.d/neo4j >> ./${LOGFILE} 2>&1
            systemctl enable neo4j.service >> ./${LOGFILE} 2>&1
            systemctl start neo4j.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo "[${NOW} install.sh] neo4j has been successfully installed." >> ./${LOGFILE}
                echo -ne "System Database installed successfully\n"
                echo -n "Please wait while the System Database is being configured. This may take several minutes..."
                until cypher-shell --username neo4j --password ${NEO4JINITIALPASS} --database system 'SHOW DATABASE system;' >> ./${LOGFILE} 2>&1
                do
                    sleep 3
                done
                sleep 3

                cypher-shell --username neo4j --password ${NEO4JINITIALPASS} --database system 'CREATE USER '${NEO4JUSER}' IF NOT EXISTS SET PASSWORD '\"${NEO4JPASS}\"' CHANGE NOT REQUIRED;' >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "System Database configured successfully\n"
                else
                    echo "[${NOW} install.sh] An error occured while trying to prepare neo4j." >> ./${LOGFILE}
                    echo -n "an error occured while trying to prepare the System Database, please refer to logfile: ${PWD}/${LOGFILE}"
                    CANCEL
                fi
            else
                echo "[${NOW} install.sh] An error occured while trying to start neo4j." >> ./${LOGFILE}
                echo -n "an error occured while trying to start the System Database, please refer to logfile: ${PWD}/${LOGFILE}"
                CANCEL
            fi
        else
            echo "[${NOW} install.sh] User has chosen not to install neo4j locally." >> ./${LOGFILE}
            echo -e "\nPlease have your login credentials on hand for the configuration at a later stage."
            PRESS_ANY_KEY
        fi
    fi
}

# Give the user the option to install through rpm (offline modus) if not already installed
CHECK_NEO4J_OFFLINE(){
    rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -e "\nThe System Database is already installed on this system. Please have your login credentials on hand for the configuration at a later stage."
        echo "[${NOW} install.sh] neo4j is already installed on this system." >> ./${LOGFILE}
    else
        EINGABE=
        while [[ ${EINGABE} != [lLeE] ]]; do
            echo -e "\n\033[0;36mThe daccord software requires a database to store data.\033[m"
            read -p $'\033[0;36mDo you want to install the System Database locally [l] or use an external [e] instance? [l/e] \033[m' EINGABE
            if [[ ${EINGABE} != [lLeE] ]]; then
                echo -e "\n\033[0;35mPlease choose between [l] to install the System Database locally or [e] to use an external instance!\033[m"
            fi
        done
        if [[ ${EINGABE} == [lL] ]]; then
            echo "[${NOW} install.sh] User has chosen to install neo4j locally." >> ./${LOGFILE}
            NEO4JINITIALPASS
            NEO4JUSER
            NEO4JPASS
            echo " "
            echo " "
            echo -n "Please wait while the System Database is being installed..."
            echo "[${NOW} install.sh] Checking if the required 'insserv-compat' package is already installed." >> ./${LOGFILE}
            rpm -qa | grep -i '^insserv-compat-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                echo "[${NOW} install.sh] The 'insserv-compat' package is already installed, proceeding with neo4j installation." >> ./${LOGFILE}
            else
                echo "[${NOW} install.sh] The 'insserv-compat' package is not yet installed, proceeding with 'insserv-compat' installation." >> ./${LOGFILE}
                rpm -Uvh insserv-compat* >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    while true; do
                        rpm -qa | grep -i '^insserv-compat-[0-9]' >> /dev/null 2>&1
                        if [[ $? == 0 ]]; then
                            sleep 3
                            break
                        fi
                        echo -n '.'
                        sleep 3
                    done
                else
                    echo "[${NOW} install.sh] Unable to install the 'insserv-compat' package" >> ./${LOGFILE}
                fi
            fi
            rpm -Uvh --nodeps cypher-shell-5.25.1-1.noarch.rpm neo4j-5.25.1-1.noarch.rpm >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                while true; do
                    rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
                    if [[ $? == 0 ]]; then
                        sleep 3
                        break
                    fi
                    echo -n '.'
                    sleep 3
                done
            else
                echo "[${NOW} install.sh] An error occured while trying to install neo4j." >> ./${LOGFILE}
                echo -n "an error occured while trying to install the System Database, please refer to logfile: ${PWD}/${LOGFILE}"
                CANCEL
            fi

            mv ./apoc-5.25.1-core.jar /var/lib/neo4j/plugins
            mv ./neo4j-daccord-event-extension-2.2.0.jar /var/lib/neo4j/plugins

            CHECK_JAVA_VERSION=$(java -version 2>&1 | grep -o --max-count=1 '"17.')
            if [[ ${CHECK_JAVA_VERSION} != '"17.' ]]; then
                update-alternatives --set java /usr/lib64/jvm/jre-17-openjdk/bin/java >> ./${LOGFILE} 2>&1
            fi
            sed -i "s/#server.default_listen_address/server.default_listen_address/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/#initial.dbms.default_database=neo4j/initial.dbms.default_database=daccord.db/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/dbms.connector.https.enabled=true/dbms.connector.https.enabled=false/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/#dbms.security.procedures.unrestricted=my.extensions.example,my.procedures.*/dbms.security.procedures.unrestricted=algo.*,apoc.*/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/#dbms.usage_report.enabled/dbms.usage_report.enabled/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1
            sed -i "s/# server.jvm.additional=-XX:+ExitOnOutOfMemoryError/server.jvm.additional=-XX:+ExitOnOutOfMemoryError/g" /etc/neo4j/neo4j.conf >> ./${LOGFILE} 2>&1

            echo "apoc.export.file.enabled=true" > /etc/neo4j/apoc.conf
            echo "apoc.import.file.enabled=true" >> /etc/neo4j/apoc.conf
            echo 'dbms.jvm.additional=-Djava.net.preferIPv4Stack=true' >> /etc/neo4j/neo4j.conf

            neo4j-admin dbms set-initial-password ${NEO4JINITIALPASS} >> ./${LOGFILE} 2>&1

            rm -f /etc/init.d/neo4j >> ./${LOGFILE} 2>&1
            systemctl enable neo4j.service >> ./${LOGFILE} 2>&1
            systemctl start neo4j.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo "[${NOW} install.sh] neo4j has been successfully installed." >> ./${LOGFILE}
                echo -ne "System Database installed successfully\n"
                echo -n "Please wait while the System Database is being configured. This may take several minutes..."
                until cypher-shell --username neo4j --password ${NEO4JINITIALPASS} --database system 'SHOW DATABASE system;' >> ./${LOGFILE} 2>&1
                do
                    sleep 3
                done
                sleep 3

                cypher-shell --username neo4j --password ${NEO4JINITIALPASS} --database system 'CREATE USER '${NEO4JUSER}' IF NOT EXISTS SET PASSWORD '\"${NEO4JPASS}\"' CHANGE NOT REQUIRED;' >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "System Database configured successfully\n"
                    echo -e "\nPlease note that a user 'daccord' with the associated password 'daccord' has been created."
                else
                    echo "[${NOW} install.sh] An error occured while trying to prepare neo4j." >> ./${LOGFILE}
                    echo -n "an error occured while try_ing to prepare the System Database, please refer to logfile: ${PWD}/${LOGFILE}"
                    CANCEL
                fi
            else
                echo "[${NOW} install.sh] An error occured while trying to start neo4j." >> ./${LOGFILE}
                echo -n "an error occured while trying to start the System Database, please refer to logfile: ${PWD}/${LOGFILE}"
                CANCEL
            fi
        else
            echo "[${NOW} install.sh] User has chosen not to install neo4j locally." >> ./${LOGFILE}
            echo -e "\nPlease have your login credentials on hand for the configuration at a later stage."
            PRESS_ANY_KEY
        fi
    fi
}

ONLINE_INSTALL(){
    echo "[${NOW} install.sh] User has requested to start the online installation." >> ./${LOGFILE}
    EINGABE=
    while [[ ${EINGABE} != [sSaA] ]]; do
        echo -e '\n\033[0;36mTo simplify the installation of the daccord components, you can now choose between a "Simple" installation, which will install all components at once, or an "Advanced" installation, where you can choose which components to install.\033[m'
        read -p $'\033[0;36mPlease choose between [s] for the "Simple" installation or [a] for the "Advanced" installation [s/a]: \033[m' EINGABE
        echo ""
        if [[ ${EINGABE} != [sSaA] ]]; then
            echo -e '\n\033[0;36mPlease choose between [s] for the "Simple" installation or [a] for the "Advanced" installation [s/a]: \033[m'
        fi
    done
    if [[ ${EINGABE} == [sS] ]]; then
        STANDARDINST="TRUE"
    else
        STANDARDINST="FALSE"
    fi
    INSTERROR=0
    if [[ ${REDHAT} == "TRUE" ]]; then
        ONLINE_REPO_INSTALL_REDHAT
    else
        ONLINE_REPO_INSTALL_SUSE
    fi
    if [[ $INSTERROR == "0" ]]; then
        echo -e "\n\033[0;36mThe selected daccord software components have been installed successfully!\033[m"
        if [[ ! ${DACCORD_CURRENT_STAT} = 1 ]]; then
            echo -e "\033[0;36mTo finalize your installation, please access the URL ${DAC_SYSTEM_URL} with a web-browser to configure your daccord system.\033[m"
        else
            DACCORD_URL=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2)
            echo -e "\033[0;36mTo finalize your installation, please access the URL ${DACCORD_URL}daccord/ with a web-browser to configure your daccord system.\033[m"
        fi
        echo -e "\n"
        rm -f /opt/GuH/daccord_basis_install_flag >> /dev/null
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the installation of the selected daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the installation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
    fi
}

ONLINE_REPO_INSTALL_SUSE(){
    echo "[${NOW} install.sh] Installing the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper lr daccord-2.2-dev >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-2.2-dev >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-2.2-dev >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-2.2-dev daccord-2.2-dev >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        zypper lr daccord-2.2 >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-2.2 >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-2.2 >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-2.2 daccord-2.2 >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo "[${NOW} install.sh] User has chosen to install the Runtime Engine." >> ./${LOGFILE}
        echo -n "Please wait while the Runtime Engine is being installed..."
        zypper --non-interactive in RuntimeEngine >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Runtime Engine installed successfully\n"
            echo "[${NOW} install.sh] RuntimeEngine installed successfully." >> ./${LOGFILE}
            SELFSIGNEDCERT
            rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
cat <<EOF> /opt/GuH/RuntimeEngine/bin/checkneo4j.sh
#!/bin/bash
until wget http://localhost:7474/ >> /dev/null 2>&1
do
    echo "Checking local neo4j interface!" >> /var/opt/GuH/RuntimeEngine/logs/catalina.out
    sleep 3
done
echo "Reached local neo4j interface!" >> /var/opt/GuH/RuntimeEngine/logs/catalina.out
rm -f ./index.html
sleep 3
exit
EOF
                chmod +x /opt/GuH/RuntimeEngine/bin/checkneo4j.sh >> ./${LOGFILE} 2>&1
                sed -i 's@PIDFile=/run/RuntimeEngine.pid@PIDFile=/run/RuntimeEngine.pid\nExecStartPre=/opt/GuH/RuntimeEngine/bin/checkneo4j.sh@g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=neo4j.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                systemctl daemon-reload >> ./${LOGFILE} 2>&1
            fi
        else
            INSTERROR=1
            echo -ne "Error while installing the Runtime Engine\n"
            echo "[${NOW} install.sh] Error while installing the RuntimeEngine." >> ./${LOGFILE}
        fi
    fi
    if [[ ${SERVERTYPE} == [pP] ]]; then
        rpm -qa | grep -i '^daccord-system-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the System Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the System Gadget." >> ./${LOGFILE}
            zypper --non-interactive in daccord-system-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "System Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-system-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the System Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-system-gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-portal-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            export DAC_PORTAL_PATH="daccord"
            if [[ ${STANDARDINST} == "FALSE" ]]; then
                EINGABE=
                while [[ ${EINGABE} != [dDoO] ]]; do
                    read -p $'\033[0;36mDo you want to use the default URL path "daccord" [d] or choose your own [o] path? [d/o] \033[m' EINGABE
                    if [[ ${EINGABE} != [dDoO] ]]; then
                        echo -n "\033[0;35mPlease choose between [d] to use the default URL path 'daccord' or [o] to choose your own path!\033[m"
                    fi
                done
                if [[ ${EINGABE} == [oO] ]]; then
                    DAC_PORTAL_PATH=
                    while [[ -z "${DAC_PORTAL_PATH}" ]]; do
                        read -p $'\033[0;36mPlease enter your own chosen URL path: \033[m' DAC_PORTAL_PATH
                        if [[ -z "${DAC_PORTAL_PATH}" ]]; then
                            echo -n "\033[0;35mPlease enter your own chosen URL path!\033[m"
                        fi
                    done
                    export DAC_PORTAL_PATH
                fi
            fi
            echo "DAC_PORTAL_PATH='${DAC_PORTAL_PATH}'" >> /etc/environment
            source /etc/environment >> ./${LOGFILE} 2>&1
            echo -n "Please wait while the Portal Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Portal Gadget." >> ./${LOGFILE}
            zypper --non-interactive in daccord-portal-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Portal Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-portal-gadget installed successfully." >> ./${LOGFILE}
                if [[ ${STANDARDINST} == "FALSE" ]]; then
                    if [[ ${DAC_PORTAL_PATH} != "daccord" ]]; then
                        mv /opt/GuH/RuntimeEngine/webapps/daccord.war /opt/GuH/RuntimeEngine/webapps/${DAC_PORTAL_PATH}.war >> ./${LOGFILE} 2>&1
                    fi
                fi
            else
                INSTERROR=1
                echo -ne "Error while installing the Portal Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-portal-gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-policies-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Policies Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Policies Gadget." >> ./${LOGFILE}
            zypper --non-interactive in daccord-policies-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Policies Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-policies-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Policies Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-policies-gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-reports-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Reports Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Reports Gadget." >> ./${LOGFILE}
            zypper --non-interactive in daccord-reports-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Reports Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-reports-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Reports Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-reports-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-collector-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Collector Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Collector Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Collector Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Collector Engine." >> ./${LOGFILE}
            zypper --non-interactive in daccord-collector-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Collector Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-collector-engine installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Collector Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-collector-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Collector Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-policy-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Policy Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Policy Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Policy Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Policy Engine." >> ./${LOGFILE}
            zypper --non-interactive in daccord-policy-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Policy Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-policy-engine installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Policy Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-policy-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Policy Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-event-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Event Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Event Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Event Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Event Engine." >> ./${LOGFILE}
            zypper --non-interactive in daccord-event-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Event Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-event-engine installed successfully." >> ./${LOGFILE}
                echo -n "Please wait while the Event Engine is being configured..."
                KAFKA_CONFIG_ERROR=0
                sh /opt/GuH/daccord-event-engine/res/config.sh INSTALL ${DAC_RUNTIMEENGINE_URL} >> ./${LOGFILE} 2>&1
                chmod +x /opt/GuH/daccord-event-engine/kafka/bin/* >> ./${LOGFILE} 2>&1
                KAFKA_CLUSTER_ID="$(/opt/GuH/daccord-event-engine/kafka/bin/kafka-storage random-uuid)" >> ./${LOGFILE} 2>&1
                /opt/GuH/daccord-event-engine/kafka/bin/kafka-storage format -t $KAFKA_CLUSTER_ID -c /opt/GuH/daccord-event-engine/kafka/etc/kafka/kraft/server.properties >> ./${LOGFILE} 2>&1
                systemctl enable kafka.service >> ./${LOGFILE} 2>&1
                systemctl start kafka.service >> ./${LOGFILE} 2>&1
                if [[ ! $? == 0 ]]; then
                    KAFKA_CONFIG_ERROR=1
                fi
                if [[ ${KAFKA_CONFIG_ERROR} == 0 ]]; then
                    echo -ne "Event Engine configured successfully\n"
                    echo "[${NOW} install.sh] daccord-event-engine configured successfully." >> ./${LOGFILE}
                else
                    echo -ne "Error while configuring the Event Engine\n"
                    echo "[${NOW} install.sh] Error while configuring the daccord-event-engine." >> ./${LOGFILE}
                fi
            else
                INSTERROR=1
                echo -ne "Error while installing the Event Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-event-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Event Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-report-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Report Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Report Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Report Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Report Engine." >> ./${LOGFILE}
            zypper --non-interactive in daccord-report-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Report Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-report-engine installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Report Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-report-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Report Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-csv-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the CSV Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the CSV Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-csv-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the CSV Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the CSV Connector." >> ./${LOGFILE}
                zypper --non-interactive in daccord-csv-connector >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "CSV Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-csv-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the CSV Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-csv-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the CSV Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-java-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the JAVA Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the JAVA Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-java-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the JAVA Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the JAVA Connector." >> ./${LOGFILE}
                zypper --non-interactive in daccord-java-connector >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "JAVA Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-java-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the JAVA Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-java-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the JAVA Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-jdbc-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the JDBC Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the JDBC Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-jdbc-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the JDBC Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the JDBC Connector." >> ./${LOGFILE}
                zypper --non-interactive in daccord-jdbc-connector >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "JDBC Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-jdbc-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the JDBC Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-jdbc-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the JDBC Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-ldap-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the LDAP Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the LDAP Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-ldap-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the LDAP Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the LDAP Connector." >> ./${LOGFILE}
                zypper --non-interactive in daccord-ldap-connector >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "LDAP Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-ldap-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the LDAP Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-ldap-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the LDAP Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-script-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Script Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the Script Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-script-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the Script Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the Script Connector." >> ./${LOGFILE}
                zypper --non-interactive in daccord-script-connector >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "Script Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-script-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the Script Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-script-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Script Connector component." >> ./${LOGFILE}
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being started..."
        echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
        systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "The Runtime Engine has been successfully started\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    systemctl status RuntimeEngine.service | grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper mr -d daccord-2.2-dev >> ./${LOGFILE} 2>&1
    else
        zypper mr -d daccord-2.2 >> ./${LOGFILE} 2>&1
    fi
}

ONLINE_REPO_INSTALL_REDHAT(){
    echo "[${NOW} install.sh] Installing the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        if [[ ! -f "/etc/yum.repos.d/daccord-2.2-dev.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-2.2-dev.repo
[daccord-2.2-dev]
name=daccord Development RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-2.2-dev
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        if [[ ! -f "/etc/yum.repos.d/daccord-2.2.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-2.2.repo
[daccord-2.2]
name=daccord RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-2.2
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being installed..."
        echo "[${NOW} install.sh] User has chosen to install the Runtime Engine." >> ./${LOGFILE}
        yum install -y RuntimeEngine >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Runtime Engine installed successfully\n"
            echo "[${NOW} install.sh] RuntimeEngine installed successfully." >> ./${LOGFILE}
            SELFSIGNEDCERT
            rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
cat <<EOF> /opt/GuH/RuntimeEngine/bin/checkneo4j.sh
#!/bin/bash
until wget http://localhost:7474/ >> /dev/null 2>&1
do
    echo "Checking local neo4j interface!" >> /var/opt/GuH/RuntimeEngine/logs/catalina.out
    sleep 3
done
echo "Reached local neo4j interface!" >> /var/opt/GuH/RuntimeEngine/logs/catalina.out
rm -f ./index.html
sleep 3
exit
EOF
                chmod +x /opt/GuH/RuntimeEngine/bin/checkneo4j.sh >> ./${LOGFILE} 2>&1
                sed -i 's@PIDFile=/run/RuntimeEngine.pid@PIDFile=/run/RuntimeEngine.pid\nExecStartPre=/opt/GuH/RuntimeEngine/bin/checkneo4j.sh@g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=neo4j.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                systemctl daemon-reload >> ./${LOGFILE} 2>&1
            fi
        else
            INSTERROR=1
            echo -ne "Error while installing the Runtime Engine\n"
            echo "[${NOW} install.sh] Error while installing the RuntimeEngine." >> ./${LOGFILE}
        fi
    fi
    if [[ ${SERVERTYPE} == [pP] ]]; then
        rpm -qa | grep -i '^daccord-system-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the System Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the System Gadget." >> ./${LOGFILE}
            yum install -y daccord-system-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "System Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-system-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the System Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-system-gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-portal-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            export DAC_PORTAL_PATH="daccord"
            if [[ ${STANDARDINST} == "FALSE" ]]; then
                EINGABE=
                while [[ ${EINGABE} != [dDoO] ]]; do
                    read -p $'\033[0;36mDo you want to use the default URL path "daccord" [d] or choose your own [o] path? [d/o] \033[m' EINGABE
                    if [[ ${EINGABE} != [dDoO] ]]; then
                        echo -n "\033[0;35mPlease choose between [d] to use the default URL path 'daccord' or [o] to choose your own path!\033[m"
                    fi
                done
                if [[ ${EINGABE} == [oO] ]]; then
                    DAC_PORTAL_PATH=
                    while [[ -z "${DAC_PORTAL_PATH}" ]]; do
                        read -p $'\033[0;36mPlease enter your own chosen URL path: \033[m' DAC_PORTAL_PATH
                        if [[ -z "${DAC_PORTAL_PATH}" ]]; then
                            echo -n "\033[0;35mPlease enter your own chosen URL path!\033[m"
                        fi
                    done
                    export DAC_PORTAL_PATH
                fi
            fi
            echo "DAC_PORTAL_PATH='${DAC_PORTAL_PATH}'" >> /etc/environment
            source /etc/environment >> ./${LOGFILE} 2>&1
            echo -n "Please wait while the Portal Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Portal Gadget." >> ./${LOGFILE}
            yum install -y daccord-portal-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Portal Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-portal-gadget installed successfully." >> ./${LOGFILE}
                if [[ ${STANDARDINST} == "FALSE" ]]; then
                    if [[ ${DAC_PORTAL_PATH} != "daccord" ]]; then
                        mv /opt/GuH/RuntimeEngine/webapps/daccord.war /opt/GuH/RuntimeEngine/webapps/${DAC_PORTAL_PATH}.war >> ./${LOGFILE} 2>&1
                    fi
                fi
            else
                INSTERROR=1
                echo -ne "Error while installing the Portal Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-portal-gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-policies-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Policies Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Policies Gadget." >> ./${LOGFILE}
            yum install -y daccord-policies-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Policies Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-policies-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Policies Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-policies-gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-reports-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Reports Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Reports Gadget." >> ./${LOGFILE}
            yum install -y daccord-reports-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Reports Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-reports-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Reports Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-reports-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-collector-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Collector Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Collector Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Collector Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Collector Engine." >> ./${LOGFILE}
            yum install -y daccord-collector-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Collector Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-collector-engine installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Collector Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-collector-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Collector Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-policy-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Policy Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Policy Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Policy Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Policy Engine." >> ./${LOGFILE}
            yum install -y daccord-policy-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Policy Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-policy-engine installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Policy Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-policy-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Policy Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-event-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Event Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Event Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Event Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Event Engine." >> ./${LOGFILE}
            yum install -y daccord-event-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Event Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-event-engine installed successfully." >> ./${LOGFILE}
                echo -n "Please wait while the Event Engine is being configured..."
                KAFKA_CONFIG_ERROR=0
                sh /opt/GuH/daccord-event-engine/res/config.sh INSTALL ${DAC_RUNTIMEENGINE_URL} >> ./${LOGFILE} 2>&1
                chmod +x /opt/GuH/daccord-event-engine/kafka/bin/* >> ./${LOGFILE} 2>&1
                KAFKA_CLUSTER_ID="$(/opt/GuH/daccord-event-engine/kafka/bin/kafka-storage random-uuid)" >> ./${LOGFILE} 2>&1
                /opt/GuH/daccord-event-engine/kafka/bin/kafka-storage format -t $KAFKA_CLUSTER_ID -c /opt/GuH/daccord-event-engine/kafka/etc/kafka/kraft/server.properties >> ./${LOGFILE} 2>&1
                systemctl enable kafka.service >> ./${LOGFILE} 2>&1
                systemctl start kafka.service >> ./${LOGFILE} 2>&1
                if [[ ! $? == 0 ]]; then
                    KAFKA_CONFIG_ERROR=1
                fi
                if [[ ${KAFKA_CONFIG_ERROR} == 0 ]]; then
                    echo -ne "Event Engine configured successfully\n"
                    echo "[${NOW} install.sh] daccord-event-engine configured successfully." >> ./${LOGFILE}
                else
                    echo -ne "Error while configuring the Event Engine\n"
                    echo "[${NOW} install.sh] Error while configuring the daccord-event-engine." >> ./${LOGFILE}
                fi
            else
                INSTERROR=1
                echo -ne "Error while installing the Event Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-event-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Event Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-report-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Report Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Report Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Report Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Report Engine." >> ./${LOGFILE}
            yum install -y daccord-report-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Report Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-report-engine installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Report Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-report-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Report Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-csv-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the CSV Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the CSV Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-csv-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the CSV Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the CSV Connector." >> ./${LOGFILE}
                yum install -y daccord-csv-connector >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "CSV Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-csv-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the CSV Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-csv-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the CSV Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-java-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the JAVA Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the JAVA Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-java-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the JAVA Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the JAVA Connector." >> ./${LOGFILE}
                yum install -y daccord-java-connector >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "JAVA Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-java-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the JAVA Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-java-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the JAVA Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-jdbc-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the JDBC Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the JDBC Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-jdbc-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the JDBC Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the JDBC Connector." >> ./${LOGFILE}
                yum install -y daccord-jdbc-connector >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "JDBC Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-jdbc-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the JDBC Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-jdbc-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the JDBC Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-ldap-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the LDAP Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the LDAP Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-ldap-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the LDAP Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the LDAP Connector." >> ./${LOGFILE}
                yum install -y daccord-ldap-connector >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "LDAP Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-ldap-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the LDAP Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-ldap-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the LDAP Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-script-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Script Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the Script Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-script-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the Script Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the Script Connector." >> ./${LOGFILE}
                yum install -y daccord-script-connector >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "Script Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-script-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the Script Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-script-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Script Connector component." >> ./${LOGFILE}
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being started..."
        echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
        systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "The Runtime Engine has been successfully started\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    systemctl status RuntimeEngine.service | grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        yum config-manager --disable daccord-2.2-dev >> ./${LOGFILE} 2>&1
    else
        yum config-manager --disable daccord-2.2 >> ./${LOGFILE} 2>&1
    fi
}

OFFLINE_INSTALL(){
    echo "[${NOW} install.sh] User has requested to start the offline installation." >> ./${LOGFILE}
    EINGABE=
    while [[ ${EINGABE} != [sSaA] ]]; do
        echo -e '\n\033[0;36mTo simplify the installation of the daccord components, you can now choose between a "Simple" installation, which will install all components at once, or an "Advanced" installation, where you can choose which components to install.\033[m'
        echo ""
        read -p $'\033[0;36mPlease choose between [s] for the "Simple" installation or [a] for the "Advanced" installation [s/a]: \033[m' EINGABE
        if [[ ${EINGABE} != [sSaA] ]]; then
            echo -e '\n\033[0;36mPlease choose between [s] for the "Simple" installation or [a] for the "Advanced" installation [s/a]: \033[m'
        fi
    done
    if [[ ${EINGABE} == [sS] ]]; then
        STANDARDINST="TRUE"
    else
        STANDARDINST="FALSE"
    fi
    INSTERROR=0
    OFFLINE_RPM_INSTALL
    if [[ $INSTERROR == "0" ]]; then
        echo -e "\nThe selected daccord software components have been installed successfully!"
        if [[ ! ${DACCORD_CURRENT_STAT} = 1 ]]; then
            echo -e "\033[0;36mTo finalize your installation, please access the URL ${DAC_SYSTEM_URL} with a web-browser to configure your daccord system.\033[m"
        else
            DACCORD_URL=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2)
            echo -e "\033[0;36mTo finalize your installation, please access the URL ${DACCORD_URL}daccord/ with a web-browser to configure your daccord system.\033[m"
        fi
        echo -e "\n"
        rm -f /opt/GuH/daccord_basis_install_flag >> /dev/null
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the installation of the selected daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the installation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
    fi
}

OFFLINE_RPM_INSTALL(){
    echo "[${NOW} install.sh] Installing the daccord software components from local rpm files" >> ./${LOGFILE}
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being installed..."
        echo "[${NOW} install.sh] User has chosen to install the Runtime Engine." >> ./${LOGFILE}
        rpm -Uvh RuntimeEngine* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Runtime Engine installed successfully\n"
            echo "[${NOW} install.sh] RuntimeEngine installed successfully." >> ./${LOGFILE}
            SELFSIGNEDCERT
            rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
cat <<EOF> /opt/GuH/RuntimeEngine/bin/checkneo4j.sh
#!/bin/bash
until wget http://localhost:7474/ >> /dev/null 2>&1
do
    echo "Checking local neo4j interface!" >> /var/opt/GuH/RuntimeEngine/logs/catalina.out
    sleep 3
done
echo "Reached local neo4j interface!" >> /var/opt/GuH/RuntimeEngine/logs/catalina.out
rm -f ./index.html
sleep 3
exit
EOF
                chmod +x /opt/GuH/RuntimeEngine/bin/checkneo4j.sh >> ./${LOGFILE} 2>&1
                sed -i 's@PIDFile=/run/RuntimeEngine.pid@PIDFile=/run/RuntimeEngine.pid\nExecStartPre=/opt/GuH/RuntimeEngine/bin/checkneo4j.sh@g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=neo4j.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                systemctl daemon-reload >> ./${LOGFILE} 2>&1
            fi
        else
            INSTERROR=1
            echo -ne "Error while installing the Runtime Engine\n"
            echo "[${NOW} install.sh] Error while installing the RuntimeEngine." >> ./${LOGFILE}
        fi
    fi
    if [[ ${SERVERTYPE} == [pP] ]]; then
        rpm -qa | grep -i '^daccord-system-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the System Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the System Gadget." >> ./${LOGFILE}
            rpm -Uvh daccord-system-gadget* >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "System Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-system-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the System Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-system-gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-portal-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            export DAC_PORTAL_PATH="daccord"
            if [[ ${STANDARDINST} == "FALSE" ]]; then
                EINGABE=
                while [[ ${EINGABE} != [dDoO] ]]; do
                    read -p $'\033[0;36mDo you want to use the default URL path "daccord" [d] or choose your own [o] path? [d/o] \033[m' EINGABE
                    if [[ ${EINGABE} != [dDoO] ]]; then
                        echo -n "\033[0;35mPlease choose between [d] to use the default URL path 'daccord' or [o] to choose your own path!\033[m"
                    fi
                done
                if [[ ${EINGABE} == [oO] ]]; then
                    DAC_PORTAL_PATH=
                    while [[ -z "${DAC_PORTAL_PATH}" ]]; do
                        read -p $'\033[0;36mPlease enter your own chosen URL path: \033[m' DAC_PORTAL_PATH
                        if [[ -z "${DAC_PORTAL_PATH}" ]]; then
                            echo -n "\033[0;35mPlease enter your own chosen URL path!\033[m"
                        fi
                    done
                    export DAC_PORTAL_PATH
                fi
            fi
            echo "DAC_PORTAL_PATH='${DAC_PORTAL_PATH}'" >> /etc/environment
            source /etc/environment >> ./${LOGFILE} 2>&1
            echo -n "Please wait while the Portal Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Portal Gadget." >> ./${LOGFILE}
            rpm -Uvh daccord-portal-gadget* >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Portal Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-portal-gadget installed successfully." >> ./${LOGFILE}
                if [[ ${STANDARDINST} == "FALSE" ]]; then
                    if [[ ${DAC_PORTAL_PATH} != "daccord" ]]; then
                        mv /opt/GuH/RuntimeEngine/webapps/daccord.war /opt/GuH/RuntimeEngine/webapps/${DAC_PORTAL_PATH}.war >> ./${LOGFILE} 2>&1
                    fi
                fi
            else
                INSTERROR=1
                echo -ne "Error while installing the Portal Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-portal-gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-policies-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Policies Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Policies Gadget." >> ./${LOGFILE}
            rpm -Uvh daccord-policies-gadget* >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Policies Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-policies-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Policies Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-policies-gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-reports-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Reports Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Reports Gadget." >> ./${LOGFILE}
            rpm -Uvh daccord-reports-gadget* >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Reports Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-reports-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Reports Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-reports-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-collector-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Collector Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Collector Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Collector Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Collector Engine." >> ./${LOGFILE}
            rpm -Uvh daccord-collector-engine* >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Collector Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-collector-engine installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Collector Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-collector-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Collector Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-policy-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Policy Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Policy Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Policy Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Policy Engine." >> ./${LOGFILE}
            rpm -Uvh daccord-policy-engine* >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Policy Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-policy-engine installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Policy Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-policy-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Policy Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-event-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Event Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Event Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Event Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Event Engine." >> ./${LOGFILE}
            rpm -Uvh daccord-event-engine* >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Event Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-event-engine installed successfully." >> ./${LOGFILE}
                echo -n "Please wait while the Event Engine is being configured..."
                KAFKA_CONFIG_ERROR=0
                sh /opt/GuH/daccord-event-engine/res/config.sh INSTALL ${DAC_RUNTIMEENGINE_URL} >> ./${LOGFILE} 2>&1
                chmod +x /opt/GuH/daccord-event-engine/kafka/bin/* >> ./${LOGFILE} 2>&1
                KAFKA_CLUSTER_ID="$(/opt/GuH/daccord-event-engine/kafka/bin/kafka-storage random-uuid)" >> ./${LOGFILE} 2>&1
                /opt/GuH/daccord-event-engine/kafka/bin/kafka-storage format -t $KAFKA_CLUSTER_ID -c /opt/GuH/daccord-event-engine/kafka/etc/kafka/kraft/server.properties >> ./${LOGFILE} 2>&1
                systemctl enable kafka.service >> ./${LOGFILE} 2>&1
                systemctl start kafka.service >> ./${LOGFILE} 2>&1
                if [[ ! $? == 0 ]]; then
                    KAFKA_CONFIG_ERROR=1
                fi
                if [[ ${KAFKA_CONFIG_ERROR} == 0 ]]; then
                    echo -ne "Event Engine configured successfully\n"
                    echo "[${NOW} install.sh] daccord-event-engine configured successfully." >> ./${LOGFILE}
                else
                    echo -ne "Error while configuring the Event Engine\n"
                    echo "[${NOW} install.sh] Error while configuring the daccord-event-engine." >> ./${LOGFILE}
                fi
            else
                INSTERROR=1
                echo -ne "Error while installing the Event Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-event-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Event Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-report-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Report Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to install the Report Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Report Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Report Engine." >> ./${LOGFILE}
            rpm -Uvh daccord-report-engine* >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Report Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-report-engine installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Report Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-report-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Report Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-csv-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the CSV Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the CSV Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-csv-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the CSV Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the CSV Connector." >> ./${LOGFILE}
                rpm -Uvh daccord-csv-connector* >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "CSV Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-csv-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the CSV Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-csv-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the CSV Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-java-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the JAVA Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the JAVA Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-java-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the JAVA Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the JAVA Connector." >> ./${LOGFILE}
                rpm -Uvh daccord-java-connector* >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "JAVA Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-java-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the JAVA Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-java-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the JAVA Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-jdbc-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the JDBC Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the JDBC Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-jdbc-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the JDBC Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the JDBC Connector." >> ./${LOGFILE}
                rpm -Uvh daccord-jdbc-connector* >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "JDBC Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-jdbc-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the JDBC Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-jdbc-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the JDBC Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-ldap-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the LDAP Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the LDAP Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-ldap-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the LDAP Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the LDAP Connector." >> ./${LOGFILE}
                rpm -Uvh daccord-ldap-connector* >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "LDAP Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-ldap-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the LDAP Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-ldap-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the LDAP Connector component." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-script-connector-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Script Connector component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the Script Connector component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            rpm -qa | grep -i '^daccord-script-connector-[0-9]' >> /dev/null 2>&1
            if [[ ! $? == 0 ]]; then
                echo -n "Please wait while the Script Connector is being installed..."
                echo "[${NOW} install.sh] User has chosen to install the Script Connector." >> ./${LOGFILE}
                rpm -Uvh daccord-script-connector* >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "Script Connector installed successfully\n"
                    echo "[${NOW} install.sh] daccord-script-connector installed successfully." >> ./${LOGFILE}
                else
                    INSTERROR=1
                    echo -ne "Error while installing the Script Connector\n"
                    echo "[${NOW} install.sh] Error while installing the daccord-script-connector." >> ./${LOGFILE}
                fi
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Script Connector component." >> ./${LOGFILE}
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being started..."
        echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
        systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "The Runtime Engine has been successfully started\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    systemctl status RuntimeEngine.service | grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
}

ONLINE_UPDATE(){
    echo "[${NOW} install.sh] User has requested to start the online update." >> ./${LOGFILE}
    UPERROR=0
    echo ""
    if [[ ${REDHAT} == "TRUE" ]]; then
        ONLINE_REPO_UPDATE_REDHAT
    else
        ONLINE_REPO_UPDATE_SUSE
    fi
    if [[ $UPERROR == "0" ]]; then
        echo -e "\nThe daccord software components have been updated successfully!"
        DACCORD_URL=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2)
        echo -e "\n\033[0;36mTo finalize the update, please access the URL ${DACCORD_URL}daccord/ with a web-browser to configure your daccord system.\033[m"
        echo -e "\n"
        EXIT
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the update of the daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the update, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
        EXIT
    fi
}

ONLINE_REPO_UPDATE_SUSE(){
    echo "[${NOW} install.sh] Updating the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper lr daccord-2.2-dev >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-2.2-dev >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh --force daccord-2.2-dev >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-2.2-dev daccord-2.2-dev >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh --force daccord-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        zypper lr daccord-2.2 >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-2.2 >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh --force daccord-2.2 >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-2.2 daccord-2.2 >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh --force daccord-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -ne "ERROR! No Runtime Engine found on the system! Cancelling the update!"
        echo "[${NOW} install.sh] ERROR! No RuntimeEngine found on the system! Cancelling the update!" >> ./${LOGFILE}
        CANCEL
    fi
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        REMANUALSTOP="TRUE"
        echo "[${NOW} install.sh] The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!" >> ./${LOGFILE}
        echo -e "The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!\n"
    else
        REMANUALSTOP="FALSE"
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for component updates..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    zypper --non-interactive up -D RuntimeEngine | grep "highest available version" >> ./${LOGFILE} 2>&1
    if [[ $? == 0 ]]; then
        echo -e "The are currently no updates for the Runtime Engine\n"
        echo "[${NOW} install.sh] The are currently no updates for the RuntimeEngine." >> ./${LOGFILE}
    else
        echo -n "Please wait while the Runtime Engine is being updated..."
        zypper --non-interactive up RuntimeEngine >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Runtime Engine updated successfully\n"
            echo "[${NOW} install.sh] RuntimeEngine updated successfully." >> ./${LOGFILE}
            rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=neo4j.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                systemctl daemon-reload >> ./${LOGFILE} 2>&1
            fi
        else
            UPERROR=1
            echo -ne "Error while updating the Runtime Engine\n"
            echo "[${NOW} install.sh] Error while updating the RuntimeEngine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-system-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-system-gadget | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the System Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-system-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the System Gadget is being updated..."
            zypper --non-interactive up daccord-system-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "System Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-system-gadget updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the System Gadget\n"
                echo "[${NOW} install.sh] Error while updating the daccord-system-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-portal-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-portal-gadget | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Portal Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-portal-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Portal Gadget is being updated..."
            zypper --non-interactive up daccord-portal-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Portal Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-portal-gadget updated successfully." >> ./${LOGFILE}
                DAC_PORTAL_PATH=$(grep "DAC_PORTAL_PATH=" /etc/environment | cut -d \' -f2)
                mv /opt/GuH/RuntimeEngine/webapps/daccord.war /opt/GuH/RuntimeEngine/webapps/${DAC_PORTAL_PATH}.war >> ./${LOGFILE} 2>&1
            else
                UPERROR=1
                echo -ne "Error while updating the Portal Gadget\n"
                echo "[${NOW} install.sh] Error while updating the daccord-portal-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-policies-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-policies-gadget | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Policies Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-policies-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Policies Gadget is being updated..."
            zypper --non-interactive up daccord-policies-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Policies Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-policies-gadget updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Policies Gadget\n"
                echo "[${NOW} install.sh] Error while updating the daccord-policies-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-reports-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-reports-gadget | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Reports Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-reports-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Reports Gadget is being updated..."
            zypper --non-interactive up daccord-reports-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Reports Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-reports-gadget updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Reports Gadget\n"
                echo "[${NOW} install.sh] Error while updating the daccord-reports-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-collector-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-collector-engine | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Collector Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-collector-engine." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Collector Engine is being updated..."
            zypper --non-interactive up daccord-collector-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Collector Engine updated successfully\n"
                echo "[${NOW} install.sh] daccord-collector-engine updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Collector Engine\n"
                echo "[${NOW} install.sh] Error while updating the daccord-collector-engine." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-policy-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-policy-engine | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Policy Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-policy-engine." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Policy Engine is being updated..."
            zypper --non-interactive up daccord-policy-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Policy Engine updated successfully\n"
                echo "[${NOW} install.sh] daccord-policy-engine updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Policy Engine\n"
                echo "[${NOW} install.sh] Error while updating the daccord-policy-engine." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-event-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-event-engine | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Event Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-event-engine." >> ./${LOGFILE}
        else
            systemctl stop kafka.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 5
            fi
            echo -n "Please wait while the Event Engine is being updated..."
            zypper --non-interactive up daccord-event-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Event Engine updated successfully\n"
                echo "[${NOW} install.sh] daccord-event-engine updated successfully." >> ./${LOGFILE}
                echo -n "Please wait while the Event Engine is being configured..."
                KAFKA_CONFIG_ERROR=0
                systemctl daemon-reload
                chmod +x /opt/GuH/daccord-event-engine/kafka/bin/* >> ./${LOGFILE} 2>&1
                systemctl start kafka.service >> ./${LOGFILE} 2>&1
                if [[ ! $? == 0 ]]; then
                    KAFKA_CONFIG_ERROR=1
                fi
                if [[ ${KAFKA_CONFIG_ERROR} == 0 ]]; then
                    echo -ne "Event Engine configured successfully\n"
                    echo "[${NOW} install.sh] daccord-event-engine configured successfully." >> ./${LOGFILE}
                else
                    echo -ne "Error while configuring the Event Engine\n"
                    echo "[${NOW} install.sh] Error while configuring the daccord-event-engine." >> ./${LOGFILE}
                fi
            else
                UPERROR=1
                echo -ne "Error while updating the Event Engine\n"
                echo "[${NOW} install.sh] Error while updating the daccord-event-engine." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-report-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-report-engine | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Report Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-report-engine." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Report Engine is being updated..."
            zypper --non-interactive up daccord-report-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Report Engine updated successfully\n"
                echo "[${NOW} install.sh] daccord-report-engine updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Report Engine\n"
                echo "[${NOW} install.sh] Error while updating the daccord-report-engine." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-csv-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-csv-connector | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the CSV Connector\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-csv-connector." >> ./${LOGFILE}
        else
            echo -n "Please wait while the CSV Connector is being updated..."
            zypper --non-interactive up daccord-csv-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "CSV Connector updated successfully\n"
                echo "[${NOW} install.sh] daccord-csv-connector updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the CSV Connector\n"
                echo "[${NOW} install.sh] Error while updating the daccord-csv-connector." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-java-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-java-connector | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the JAVA Connector\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-java-connector." >> ./${LOGFILE}
        else
            echo -n "Please wait while the JAVA Connector is being updated..."
            zypper --non-interactive up daccord-java-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "JAVA Connector updated successfully\n"
                echo "[${NOW} install.sh] daccord-java-connector updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the JAVA Connector\n"
                echo "[${NOW} install.sh] Error while updating the daccord-java-connector." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-jdbc-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-jdbc-connector | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the JDBC Connector\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-jdbc-connector." >> ./${LOGFILE}
        else
            echo -n "Please wait while the JDBC Connector is being updated..."
            zypper --non-interactive up daccord-jdbc-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "JDBC Connector updated successfully\n"
                echo "[${NOW} install.sh] daccord-jdbc-connector updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the JDBC Connector\n"
                echo "[${NOW} install.sh] Error while updating the daccord-jdbc-connector." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-ldap-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-ldap-connector | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the LDAP Connector\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-ldap-connector." >> ./${LOGFILE}
        else
            echo -n "Please wait while the LDAP Connector is being updated..."
            zypper --non-interactive up daccord-ldap-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "LDAP Connector updated successfully\n"
                echo "[${NOW} install.sh] daccord-ldap-connector updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the LDAP Connector\n"
                echo "[${NOW} install.sh] Error while updating the daccord-ldap-connector." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-script-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-script-connector | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Script Connector\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-script-connector." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Script Connector is being updated..."
            zypper --non-interactive up daccord-script-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Script Connector updated successfully\n"
                echo "[${NOW} install.sh] daccord-script-connector updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Script Connector\n"
                echo "[${NOW} install.sh] Error while updating the daccord-script-connector." >> ./${LOGFILE}
            fi
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    if [[ ${REMANUALSTOP} == "FALSE" ]]; then
        systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Runtime Engine is being started..."
            echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
            systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 15 >> /dev/null 2>&1
                echo -ne "The Runtime Engine has been successfully started\n"
                echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    fi
    systemctl status RuntimeEngine.service | grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper mr -d daccord-2.2-dev >> ./${LOGFILE} 2>&1
    else
        zypper mr -d daccord-2.2 >> ./${LOGFILE} 2>&1
    fi
}

ONLINE_REPO_UPDATE_REDHAT(){
    echo "[${NOW} install.sh] Updating the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        if [[ ! -f "/etc/yum.repos.d/daccord-2.2-dev.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-2.2-dev.repo
[daccord-2.2-dev]
name=daccord dev RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-2.2-dev
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        if [[ ! -f "/etc/yum.repos.d/daccord-2.2.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-2.2.repo
[daccord-2.2]
name=daccord 2.2 RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-2.2
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    yum clean all >> /dev/null 2>&1
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -ne "ERROR! No Runtime Engine found on the system! Cancelling the update!"
        echo "[${NOW} install.sh] ERROR! No RuntimeEngine found on the system! Cancelling the update!" >> ./${LOGFILE}
        CANCEL
    fi
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        REMANUALSTOP="TRUE"
        echo "[${NOW} install.sh] The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!" >> ./${LOGFILE}
        echo -e "The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!\n"
    else
        REMANUALSTOP="FALSE"
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for component updates..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    yum check-update RuntimeEngine | grep daccord-2.2 >> ./${LOGFILE} 2>&1
    if [[ ! $? == 0 ]]; then
        echo -e "The are currently no updates for the Runtime Engine\n"
        echo "[${NOW} install.sh] The are currently no updates for the RuntimeEngine." >> ./${LOGFILE}
    else
        echo -n "Please wait while the Runtime Engine is being updated..."
        yum upgrade -y RuntimeEngine >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Runtime Engine updated successfully\n"
            echo "[${NOW} install.sh] RuntimeEngine updated successfully." >> ./${LOGFILE}
            rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=neo4j.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                systemctl daemon-reload >> ./${LOGFILE} 2>&1
            fi
        else
            UPERROR=1
            echo -ne "Error while updating the Runtime Engine\n"
            echo "[${NOW} install.sh] Error while updating the RuntimeEngine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-system-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-system-gadget | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the System Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-system-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the System Gadget is being updated..."
            yum upgrade -y daccord-system-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "System Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-system-gadget updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the System Gadget\n"
                echo "[${NOW} install.sh] Error while updating the daccord-system-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-portal-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-portal-gadget | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Portal Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-portal-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Portal Gadget is being updated..."
            yum upgrade -y daccord-portal-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Portal Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-portal-gadget updated successfully." >> ./${LOGFILE}
                DAC_PORTAL_PATH=$(grep "DAC_PORTAL_PATH=" /etc/environment | cut -d \' -f2)
                mv /opt/GuH/RuntimeEngine/webapps/daccord.war /opt/GuH/RuntimeEngine/webapps/${DAC_PORTAL_PATH}.war >> ./${LOGFILE} 2>&1
            else
                UPERROR=1
                echo -ne "Error while updating the Portal Gadget\n"
                echo "[${NOW} install.sh] Error while updating the daccord-portal-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-policies-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-policies-gadget | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Policies Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-poliices-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Policies Gadget is being updated..."
            yum upgrade -y daccord-policies-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Policies Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-policies-gadget updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Policies Gadget\n"
                echo "[${NOW} install.sh] Error while updating the daccord-policies-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-reports-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-reports-gadget | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Reports Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-reports-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Reports Gadget is being updated..."
            yum upgrade -y daccord-reports-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Reports Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-reports-gadget updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Reports Gadget\n"
                echo "[${NOW} install.sh] Error while updating the daccord-reports-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-collector-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-collector-engine | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Collector Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-collector-engine." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Collector Engine is being updated..."
            yum upgrade -y daccord-collector-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Collector Engine updated successfully\n"
                echo "[${NOW} install.sh] daccord-collector-engine updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Collector Engine\n"
                echo "[${NOW} install.sh] Error while updating the daccord-collector-engine." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-policy-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-policy-engine | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Policy Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-policy-engine." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Policy Engine is being updated..."
            yum upgrade -y daccord-policy-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Policy Engine updated successfully\n"
                echo "[${NOW} install.sh] daccord-policy-engine updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Policy Engine\n"
                echo "[${NOW} install.sh] Error while updating the daccord-policy-engine." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-event-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-event-engine | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Event Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-event-engine." >> ./${LOGFILE}
        else
            systemctl stop kafka.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 5
            fi
            echo -n "Please wait while the Event Engine is being updated..."
            yum upgrade -y daccord-event-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Event Engine updated successfully\n"
                echo "[${NOW} install.sh] daccord-event-engine updated successfully." >> ./${LOGFILE}
                echo -n "Please wait while the Event Engine is being configured..."
                KAFKA_CONFIG_ERROR=0
                systemctl daemon-reload
                chmod +x /opt/GuH/daccord-event-engine/kafka/bin/* >> ./${LOGFILE} 2>&1
                systemctl start kafka.service >> ./${LOGFILE} 2>&1
                if [[ ! $? == 0 ]]; then
                    KAFKA_CONFIG_ERROR=1
                fi
                if [[ ${KAFKA_CONFIG_ERROR} == 0 ]]; then
                    echo -ne "Event Engine configured successfully\n"
                    echo "[${NOW} install.sh] daccord-event-engine configured successfully." >> ./${LOGFILE}
                else
                    echo -ne "Error while configuring the Event Engine\n"
                    echo "[${NOW} install.sh] Error while configuring the daccord-event-engine." >> ./${LOGFILE}
                fi
            else
                UPERROR=1
                echo -ne "Error while updating the Event Engine\n"
                echo "[${NOW} install.sh] Error while updating the daccord-event-engine." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-report-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-report-engine | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Report Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-report-engine." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Report Engine is being updated..."
            yum upgrade -y daccord-report-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Report Engine updated successfully\n"
                echo "[${NOW} install.sh] daccord-report-engine updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Report Engine\n"
                echo "[${NOW} install.sh] Error while updating the daccord-report-engine." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-csv-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-csv-connector | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the CSV Connector\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-csv-connector." >> ./${LOGFILE}
        else
            echo -n "Please wait while the CSV Connector is being updated..."
            yum upgrade -y daccord-csv-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "CSV Connector updated successfully\n"
                echo "[${NOW} install.sh] daccord-csv-connector updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the CSV Connector\n"
                echo "[${NOW} install.sh] Error while updating the daccord-csv-connector." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-java-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-java-connector | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the JAVA Connector\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-java-connector." >> ./${LOGFILE}
        else
            echo -n "Please wait while the JAVA Connector is being updated..."
            yum upgrade -y daccord-java-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "JAVA Connector updated successfully\n"
                echo "[${NOW} install.sh] daccord-java-connector updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the JAVA Connector\n"
                echo "[${NOW} install.sh] Error while updating the daccord-java-connector." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-jdbc-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-jdbc-connector | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the JDBC Connector\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-jdbc-connector." >> ./${LOGFILE}
        else
            echo -n "Please wait while the JDBC Connector is being updated..."
            yum upgrade -y daccord-jdbc-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "JDBC Connector updated successfully\n"
                echo "[${NOW} install.sh] daccord-jdbc-connector updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the JDBC Connector\n"
                echo "[${NOW} install.sh] Error while updating the daccord-jdbc-connector." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-ldap-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-ldap-connector | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the LDAP Connector\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-ldap-connector." >> ./${LOGFILE}
        else
            echo -n "Please wait while the LDAP Connector is being updated..."
            yum upgrade -y daccord-ldap-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "LDAP Connector updated successfully\n"
                echo "[${NOW} install.sh] daccord-ldap-connector updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the LDAP Connector\n"
                echo "[${NOW} install.sh] Error while updating the daccord-ldap-connector." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-script-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-script-connector | grep daccord-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Script Connector\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-script-connector." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Script Connector is being updated..."
            yum upgrade -y daccord-script-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Script Connector updated successfully\n"
                echo "[${NOW} install.sh] daccord-script-connector updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Script Connector\n"
                echo "[${NOW} install.sh] Error while updating the daccord-script-connector." >> ./${LOGFILE}
            fi
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    if [[ ${REMANUALSTOP} == "FALSE" ]]; then
        systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Runtime Engine is being started..."
            echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
            systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 15 >> /dev/null 2>&1
                echo -ne "The Runtime Engine has been successfully started\n"
                echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    fi
    systemctl status RuntimeEngine.service | grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        yum config-manager --disable daccord-2.2-dev >> ./${LOGFILE} 2>&1
    else
        yum config-manager --disable daccord-2.2 >> ./${LOGFILE} 2>&1
    fi
}

OFFLINE_UPDATE(){
    echo "[${NOW} install.sh] User has requested to start the offline update." >> ./${LOGFILE}
    UPERROR=0
    echo ""
    OFFLINE_RPM_UPDATE
    if [[ $UPERROR == "0" ]]; then
        echo -e "\nThe daccord software components have been updated successfully!"
        DACCORD_URL=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2)
        echo -e "\n\033[0;36mTo finalize the update, please access the URL ${DACCORD_URL}daccord/ with a web-browser to configure your daccord system.\033[m"
        echo -e "\n"
        EXIT
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the update of the daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the update, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
        EXIT
    fi
}

OFFLINE_RPM_UPDATE(){
    echo "[${NOW} install.sh] Updating the daccord software components from local rpm files." >> ./${LOGFILE}
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -ne "ERROR! No Runtime Engine found on the system! Cancelling the update!"
        echo "[${NOW} install.sh] ERROR! No RuntimeEngine found on the system! Cancelling the update!" >> ./${LOGFILE}
        CANCEL
    fi
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        REMANUALSTOP="TRUE"
        echo "[${NOW} install.sh] The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!" >> ./${LOGFILE}
        echo -e "The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!\n"
    else
        REMANUALSTOP="FALSE"
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for component updates..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    echo -n "Please wait while the Runtime Engine is being updated..."
    rpm -Uvh RuntimeEngine* >> ./${LOGFILE} 2>&1
    if [[ $? == 0 ]]; then
        echo -ne "Runtime Engine updated successfully\n"
        echo "[${NOW} install.sh] RuntimeEngine updated successfully." >> ./${LOGFILE}
        rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=neo4j.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
            systemctl daemon-reload >> ./${LOGFILE} 2>&1
        fi
    else
        UPERROR=1
        echo -ne "Error while updating the Runtime Engine\n"
        echo "[${NOW} install.sh] Error while updating the RuntimeEngine." >> ./${LOGFILE}
    fi
    rpm -qa | grep -i '^daccord-system-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the System Gadget is being updated..."
        rpm -Uvh daccord-system-gadget* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "System Gadget updated successfully\n"
            echo "[${NOW} install.sh] daccord-system-gadget updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the System Gadget\n"
            echo "[${NOW} install.sh] Error while updating the daccord-system-gadget." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-portal-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Portal Gadget is being updated..."
        rpm -Uvh daccord-portal-gadget* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Portal Gadget updated successfully\n"
            echo "[${NOW} install.sh] daccord-portal-gadget updated successfully." >> ./${LOGFILE}
            DAC_PORTAL_PATH=$(grep "DAC_PORTAL_PATH=" /etc/environment | cut -d \' -f2)
            mv /opt/GuH/RuntimeEngine/webapps/daccord.war /opt/GuH/RuntimeEngine/webapps/${DAC_PORTAL_PATH}.war >> ./${LOGFILE} 2>&1
        else
            UPERROR=1
            echo -ne "Error while updating the Portal Gadget\n"
            echo "[${NOW} install.sh] Error while updating the daccord-portal-gadget." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-policies-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Policies Gadget is being updated..."
        rpm -Uvh daccord-policies-gadget* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Policies Gadget updated successfully\n"
            echo "[${NOW} install.sh] daccord-policies-gadget updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the Policies Gadget\n"
            echo "[${NOW} install.sh] Error while updating the daccord-policies-gadget." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-reports-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Reports Gadget is being updated..."
        rpm -Uvh daccord-reports-gadget* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Reports Gadget updated successfully\n"
            echo "[${NOW} install.sh] daccord-reports-gadget updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the Reports Gadget\n"
            echo "[${NOW} install.sh] Error while updating the daccord-reports-gadget." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-collector-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Collector Engine is being updated..."
        rpm -Uvh daccord-collector-engine* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Collector Engine updated successfully\n"
            echo "[${NOW} install.sh] daccord-collector-engine updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the Collector Engine\n"
            echo "[${NOW} install.sh] Error while updating the daccord-collector-engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-policy-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Policy Engine is being updated..."
        rpm -Uvh daccord-policy-engine* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Policy Engine updated successfully\n"
            echo "[${NOW} install.sh] daccord-policy-engine updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the Policy Engine\n"
            echo "[${NOW} install.sh] Error while updating the daccord-policy-engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-event-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        systemctl stop kafka.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 5
        fi
        echo -n "Please wait while the Event Engine is being updated..."
        rpm -Uvh daccord-event-engine* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Event Engine updated successfully\n"
            echo "[${NOW} install.sh] daccord-event-engine updated successfully." >> ./${LOGFILE}
            echo -n "Please wait while the Event Engine is being configured..."
            KAFKA_CONFIG_ERROR=0
            systemctl daemon-reload
            chmod +x /opt/GuH/daccord-event-engine/kafka/bin/* >> ./${LOGFILE} 2>&1
            systemctl start kafka.service >> ./${LOGFILE} 2>&1
            if [[ ! $? == 0 ]]; then
                KAFKA_CONFIG_ERROR=1
            fi
            if [[ ${KAFKA_CONFIG_ERROR} == 0 ]]; then
                echo -ne "Event Engine configured successfully\n"
                echo "[${NOW} install.sh] daccord-event-engine configured successfully." >> ./${LOGFILE}
            else
                echo -ne "Error while configuring the Event Engine\n"
                echo "[${NOW} install.sh] Error while configuring the daccord-event-engine." >> ./${LOGFILE}
            fi
        else
            UPERROR=1
            echo -ne "Error while updating the Event Engine\n"
            echo "[${NOW} install.sh] Error while updating the daccord-event-engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-report-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Report Engine is being updated..."
        rpm -Uvh daccord-report-engine* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Report Engine updated successfully\n"
            echo "[${NOW} install.sh] daccord-report-engine updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the Report Engine\n"
            echo "[${NOW} install.sh] Error while updating the daccord-report-engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-csv-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the CSV Connector is being updated..."
        rpm -Uvh daccord-csv-connector* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "CSV Connector updated successfully\n"
            echo "[${NOW} install.sh] daccord-csv-connector updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the CSV Connector\n"
            echo "[${NOW} install.sh] Error while updating the daccord-csv-connector." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-java-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the JAVA Connector is being updated..."
        rpm -Uvh daccord-java-connector* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "JAVA Connector updated successfully\n"
            echo "[${NOW} install.sh] daccord-java-connector updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the JAVA Connector\n"
            echo "[${NOW} install.sh] Error while updating the daccord-java-connector." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-jdbc-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the JDBC Connector is being updated..."
        rpm -Uvh daccord-jdbc-connector* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "JDBC Connector updated successfully\n"
            echo "[${NOW} install.sh] daccord-jdbc-connector updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the JDBC Connector\n"
            echo "[${NOW} install.sh] Error while updating the daccord-jdbc-connector." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-ldap-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the LDAP Connector is being updated..."
        rpm -Uvh daccord-ldap-connector* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "LDAP Connector updated successfully\n"
            echo "[${NOW} install.sh] daccord-ldap-connector updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the LDAP Connector\n"
            echo "[${NOW} install.sh] Error while updating the daccord-ldap-connector." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-script-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Script Connector is being updated..."
        rpm -Uvh daccord-script-connector* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Script Connector updated successfully\n"
            echo "[${NOW} install.sh] daccord-script-connector updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the Script Connector\n"
            echo "[${NOW} install.sh] Error while updating the daccord-script-connector." >> ./${LOGFILE}
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    if [[ ${REMANUALSTOP} == "FALSE" ]]; then
        systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Runtime Engine is being started..."
            echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
            systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 15 >> /dev/null 2>&1
                echo -ne "The Runtime Engine has been successfully started\n"
                echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    if [[ ${REMANUALSTOP} == "FALSE" ]]; then
        systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Runtime Engine is being started..."
            echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
            systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 15 >> /dev/null 2>&1
                echo -ne "The Runtime Engine has been successfully started\n"
                echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    fi
    systemctl status RuntimeEngine.service | grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
}

UNINSTALL(){
    DEINSTERROR=0
    echo "[${NOW} install.sh] User has requested to start the deinstallation." >> ./${LOGFILE}
    EINGABE=
    while [[ ${EINGABE} != [sSaA] ]]; do
        echo -e '\n\033[0;36mTo simplify the deinstallation of the daccord components, you can now choose between a "Simple" deinstallation, which will uninstall all components at once, or an "Advanced" deinstallation, where you can choose which components to uninstall.\033[m'
        read -p $'\033[0;36mPlease choose between [s] for the "Simple" deinstallation or [a] for the "Advanced" deinstallation [s/a]: \033[m' EINGABE
        echo ""
        if [[ ${EINGABE} != [sSaA] ]]; then
            echo -e '\n\033[0;36mPlease choose between [s] for the "Simple" deinstallation or [a] for the "Advanced" deinstallation [s/a]: \033[m'
        fi
    done
    if [[ ${EINGABE} == [sS] ]]; then
        STANDARDDEINST="TRUE"
    else
        STANDARDDEINST="FALSE"
    fi
    if [[ ${REDHAT} == "TRUE" ]]; then
        if [[ -f "/etc/yum.repos.d/daccord-2.2-dev.repo" ]]; then
            rm -f /etc/yum.repos.d/daccord-2.2-dev.repo >> ./${LOGFILE} 2>&1
        fi
        if [[ -f "/etc/yum.repos.d/daccord-2.2.repo" ]]; then
            rm -f /etc/yum.repos.d/daccord-2.2.repo >> ./${LOGFILE} 2>&1
        fi
    else
        zypper lr daccord-2.2-dev >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper rr daccord-2.2-dev >> ./${LOGFILE} 2>&1
        fi
        zypper lr daccord-2.2 >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper rr daccord-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for component deinstallations..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    rpm -qa | grep -i '^daccord-script-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        if [[ ${STANDARDDEINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to uninstall the Script Connector? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to uninstall the Script Connector or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Script Connector is being uninstalled..."
            echo "[${NOW} install.sh] User has chosen to uninstall the Script Connector." >> ./${LOGFILE}
            rpm -ve daccord-script-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Script Connector uninstalled successfully\n"
                echo "[${NOW} install.sh] daccord-script-connector uninstalled successfully." >> ./${LOGFILE}
                rm -fR "/opt/GuH/daccord-script-connector" "/var/opt/GuH/daccord-script-connector" "/opt/GuH/RuntimeEngine/webapps/scriptconnector" "/opt/GuH/RuntimeEngine/webapps/scriptconnector.war" >> ./${LOGFILE} 2>&1
            else
                DEINSTERROR=1
                echo -ne "Error while uninstalling the Script Connector\n"
                echo "[${NOW} install.sh] Error while uninstalling the daccord-script-connector." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Script Connector." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-ldap-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        if [[ ${STANDARDDEINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to uninstall the LDAP Connector? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to uninstall the LDAP Connector or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the LDAP Connector is being uninstalled..."
            echo "[${NOW} install.sh] User has chosen to uninstall the LDAP Connector." >> ./${LOGFILE}
            rpm -ve daccord-ldap-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "LDAP Connector uninstalled successfully\n"
                echo "[${NOW} install.sh] daccord-ldap-connector uninstalled successfully." >> ./${LOGFILE}
                rm -fR "/opt/GuH/daccord-ldap-connector" "/var/opt/GuH/daccord-ldap-connector" "/opt/GuH/RuntimeEngine/webapps/ldapconnector" "/opt/GuH/RuntimeEngine/webapps/ldapconnector.war" >> ./${LOGFILE} 2>&1
            else
                DEINSTERROR=1
                echo -ne "Error while uninstalling the LDAP Connector\n"
                echo "[${NOW} install.sh] Error while uninstalling the daccord-ldap-connector." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Script Connector." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-jdbc-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        if [[ ${STANDARDDEINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to uninstall the JDBC Connector? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to uninstall the JDBC Connector or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the JDBC Connector is being uninstalled..."
            echo "[${NOW} install.sh] User has chosen to uninstall the JDBC Connector." >> ./${LOGFILE}
            rpm -ve daccord-jdbc-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "JDBC Connector uninstalled successfully\n"
                echo "[${NOW} install.sh] daccord-jdbc-connector uninstalled successfully." >> ./${LOGFILE}
                rm -fR "/opt/GuH/daccord-jdbc-connector" "/var/opt/GuH/daccord-jdbc-connector" "/opt/GuH/RuntimeEngine/webapps/jdbcconnector" "/opt/GuH/RuntimeEngine/webapps/jdbcconnector.war" >> ./${LOGFILE} 2>&1
            else
                DEINSTERROR=1
                echo -ne "Error while uninstalling the JDBC Connector\n"
                echo "[${NOW} install.sh] Error while uninstalling the daccord-jdbc-connector." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the JDBC Connector." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-java-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        if [[ ${STANDARDDEINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to uninstall the JAVA Connector? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to uninstall the JAVA Connector or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the JAVA Connector is being uninstalled..."
            echo "[${NOW} install.sh] User has chosen to uninstall the JAVA Connector." >> ./${LOGFILE}
            rpm -ve daccord-java-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "JAVA Connector uninstalled successfully\n"
                echo "[${NOW} install.sh] daccord-java-connector uninstalled successfully." >> ./${LOGFILE}
                rm -fR "/opt/GuH/daccord-java-connector" "/var/opt/GuH/daccord-java-connector" "/opt/GuH/RuntimeEngine/webapps/javaconnector" "/opt/GuH/RuntimeEngine/webapps/javaconnector.war" >> ./${LOGFILE} 2>&1
            else
                DEINSTERROR=1
                echo -ne "Error while uninstalling the JAVA Connector\n"
                echo "[${NOW} install.sh] Error while uninstalling the daccord-java-connector." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the JAVA Connector." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-csv-connector-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        if [[ ${STANDARDDEINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to uninstall the CSV Connector? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to uninstall the CSV Connector or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the CSV Connector is being uninstalled..."
            echo "[${NOW} install.sh] User has chosen to uninstall the CSV Connector." >> ./${LOGFILE}
            rpm -ve daccord-csv-connector >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "CSV Connector uninstalled successfully\n"
                echo "[${NOW} install.sh] daccord-csv-connector uninstalled successfully." >> ./${LOGFILE}
                rm -fR "/opt/GuH/daccord-csv-connector" "/var/opt/GuH/daccord-csv-connector" "/opt/GuH/RuntimeEngine/webapps/csvconnector" "/opt/GuH/RuntimeEngine/webapps/csvconnector.war" >> ./${LOGFILE} 2>&1
            else
                DEINSTERROR=1
                echo -ne "Error while uninstalling the CSV Connector\n"
                echo "[${NOW} install.sh] Error while uninstalling the daccord-csv-connector." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the CSV Connector." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-report-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        if [[ ${STANDARDDEINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to uninstall the Report Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to uninstall the Report Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Report Engine is being uninstalled..."
            echo "[${NOW} install.sh] User has chosen to uninstall the Report Engine." >> ./${LOGFILE}
            rpm -ve daccord-report-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Report Engine uninstalled successfully\n"
                echo "[${NOW} install.sh] daccord-report-engine uninstalled successfully." >> ./${LOGFILE}
                rm -fR "/opt/GuH/daccord-report-engine" "/var/opt/GuH/daccord-report-engine" "/opt/GuH/RuntimeEngine/webapps/reportengine" "/opt/GuH/RuntimeEngine/webapps/reportengine.war" >> ./${LOGFILE} 2>&1
            else
                DEINSTERROR=1
                echo -ne "Error while uninstalling the Report Engine\n"
                echo "[${NOW} install.sh] Error while uninstalling the daccord-report-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Report Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-event-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        if [[ ${STANDARDDEINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to uninstall the Event Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to uninstall the Event Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Event Engine is being uninstalled..."
            echo "[${NOW} install.sh] User has chosen to uninstall the Event Engine." >> ./${LOGFILE}
            rpm -ve daccord-event-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Event Engine uninstalled successfully\n"
                echo "[${NOW} install.sh] daccord-event-engine uninstalled successfully." >> ./${LOGFILE}
                rm -fR "/opt/GuH/daccord-event-engine" "/var/opt/GuH/daccord-event-engine" "/opt/GuH/RuntimeEngine/webapps/eventengine" "/opt/GuH/RuntimeEngine/webapps/eventengine.war" >> ./${LOGFILE} 2>&1
            else
                DEINSTERROR=1
                echo -ne "Error while uninstalling the Event Engine\n"
                echo "[${NOW} install.sh] Error while uninstalling the daccord-event-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Event Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-policy-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        if [[ ${STANDARDDEINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to uninstall the Policy Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to uninstall the Policy Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Policy Engine is being uninstalled..."
            echo "[${NOW} install.sh] User has chosen to uninstall the Policy Engine." >> ./${LOGFILE}
            rpm -ve daccord-policy-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Policy Engine uninstalled successfully\n"
                echo "[${NOW} install.sh] daccord-policy-engine uninstalled successfully." >> ./${LOGFILE}
                rm -fR "/opt/GuH/daccord-policy-engine" "/var/opt/GuH/daccord-policy-engine" "/opt/GuH/RuntimeEngine/webapps/policyengine" "/opt/GuH/RuntimeEngine/webapps/policyengine.war" >> ./${LOGFILE} 2>&1
            else
                DEINSTERROR=1
                echo -ne "Error while uninstalling the Policy Engine\n"
                echo "[${NOW} install.sh] Error while uninstalling the daccord-policy-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Policy Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-collector-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        if [[ ${STANDARDDEINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to uninstall the Collector Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to uninstall the Collector Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Collector Engine is being uninstalled..."
            echo "[${NOW} install.sh] User has chosen to uninstall the Collector Engine." >> ./${LOGFILE}
            rpm -ve daccord-collector-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Collector Engine uninstalled successfully\n"
                echo "[${NOW} install.sh] daccord-collector-engine uninstalled successfully." >> ./${LOGFILE}
                rm -fR "/opt/GuH/daccord-collector-engine" "/var/opt/GuH/daccord-collector-engine" "/opt/GuH/RuntimeEngine/webapps/collectorengine" "/opt/GuH/RuntimeEngine/webapps/collectorengine.war" >> ./${LOGFILE} 2>&1
            else
                DEINSTERROR=1
                echo -ne "Error while uninstalling the Collector Engine\n"
                echo "[${NOW} install.sh] Error while uninstalling the daccord-collector-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Collector Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep daccord | grep -vE "daccord-system-gadget|daccord-portal-gadget|daccord-policies-gadget|daccord-reports-gadget" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        rpm -qa | grep -i '^daccord-reports-gadget-[0-9]' >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            if [[ ${STANDARDDEINST} == "TRUE" ]]; then
                EINGABE="y"
            else
                EINGABE=
            fi
            while [[ ${EINGABE} != [yYnN] ]]; do
                read -p $'\033[0;36mDo you want to uninstall the Reports Gadget? [y/n] \033[m' EINGABE
                if [[ ${EINGABE} != [yYnN] ]]; then
                    echo -n "\033[0;35mPlease choose between [y] to uninstall the Reports Gadget or [n] to skip this!\033[m"
                fi
            done
            if [[ ${EINGABE} == [yY] ]]; then
                echo -n "Please wait while the Reports Gadget is being uninstalled..."
                echo "[${NOW} install.sh] User has chosen to uninstall the Reports Gadget." >> ./${LOGFILE}
                rpm -ve daccord-reports-gadget >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "Reports Gadget uninstalled successfully\n"
                    echo "[${NOW} install.sh] daccord-reports-gadget uninstalled successfully." >> ./${LOGFILE}
                    rm -fR "/opt/GuH/daccord-reports-gadget" "/var/opt/GuH/daccord-reports-gadget" "/opt/GuH/RuntimeEngine/webapps/reports" "/opt/GuH/RuntimeEngine/webapps/reports.war" >> ./${LOGFILE} 2>&1
                else
                    DEINSTERROR=1
                    echo -ne "Error while uninstalling the Reports Gadget\n"
                    echo "[${NOW} install.sh] Error while uninstalling the daccord-reports-gadget." >> ./${LOGFILE}
                fi
            else
                echo "[${NOW} install.sh] User has chosen to skip the Reports Gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-policies-gadget-[0-9]' >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            if [[ ${STANDARDDEINST} == "TRUE" ]]; then
                EINGABE="y"
            else
                EINGABE=
            fi
            while [[ ${EINGABE} != [yYnN] ]]; do
                read -p $'\033[0;36mDo you want to uninstall the Policies Gadget? [y/n] \033[m' EINGABE
                if [[ ${EINGABE} != [yYnN] ]]; then
                    echo -n "\033[0;35mPlease choose between [y] to uninstall the Policies Gadget or [n] to skip this!\033[m"
                fi
            done
            if [[ ${EINGABE} == [yY] ]]; then
                echo -n "Please wait while the Policies Gadget is being uninstalled..."
                echo "[${NOW} install.sh] User has chosen to uninstall the Policies Gadget." >> ./${LOGFILE}
                rpm -ve daccord-policies-gadget >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "Policies Gadget uninstalled successfully\n"
                    echo "[${NOW} install.sh] daccord-policies-gadget uninstalled successfully." >> ./${LOGFILE}
                    rm -fR "/opt/GuH/daccord-policies-gadget" "/var/opt/GuH/daccord-policies-gadget" "/opt/GuH/RuntimeEngine/webapps/policies" "/opt/GuH/RuntimeEngine/webapps/policies.war" >> ./${LOGFILE} 2>&1
                else
                    DEINSTERROR=1
                    echo -ne "Error while uninstalling the Policies Gadget\n"
                    echo "[${NOW} install.sh] Error while uninstalling the daccord-policies-gadget." >> ./${LOGFILE}
                fi
            else
                echo "[${NOW} install.sh] User has chosen to skip the Policies Gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-portal-gadget-[0-9]' >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            if [[ ${STANDARDDEINST} == "TRUE" ]]; then
                EINGABE="y"
            else
                EINGABE=
            fi
            while [[ ${EINGABE} != [yYnN] ]]; do
                read -p $'\033[0;36mDo you want to uninstall the Portal Gadget? [y/n] \033[m' EINGABE
                if [[ ${EINGABE} != [yYnN] ]]; then
                    echo -n "\033[0;35mPlease choose between [y] to uninstall the Portal Gadget or [n] to skip this!\033[m"
                fi
            done
            if [[ ${EINGABE} == [yY] ]]; then
                echo -n "Please wait while the Portal Gadget is being uninstalled..."
                echo "[${NOW} install.sh] User has chosen to uninstall the Portal Gadget." >> ./${LOGFILE}
                rpm -ve daccord-portal-gadget >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "Portal Gadget uninstalled successfully\n"
                    echo "[${NOW} install.sh] daccord-portal-gadget uninstalled successfully." >> ./${LOGFILE}
                    DAC_PORTAL_PATH=$(grep "DAC_PORTAL_PATH=" /etc/environment | cut -d \' -f2)
                    rm -fR "/opt/GuH/daccord-portal-gadget" "/var/opt/GuH/daccord-portal-gadget" "/opt/GuH/RuntimeEngine/webapps/${DAC_PORTAL_PATH}" "/opt/GuH/RuntimeEngine/webapps/${DAC_PORTAL_PATH}.war" >> ./${LOGFILE} 2>&1
                else
                    DEINSTERROR=1
                    echo -ne "Error while uninstalling the Portal Gadget\n"
                    echo "[${NOW} install.sh] Error while uninstalling the daccord-portal-gadget." >> ./${LOGFILE}
                fi
            else
                echo "[${NOW} install.sh] User has chosen to skip the Portal Gadget." >> ./${LOGFILE}
            fi
        fi
        rpm -qa | grep -i '^daccord-system-gadget-[0-9]' >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            if [[ ${STANDARDDEINST} == "TRUE" ]]; then
                EINGABE="y"
            else
                EINGABE=
            fi
            while [[ ${EINGABE} != [yYnN] ]]; do
                read -p $'\033[0;36mDo you want to uninstall the System Gadget? [y/n] \033[m' EINGABE
                if [[ ${EINGABE} != [yYnN] ]]; then
                    echo -n "\033[0;35mPlease choose between [y] to uninstall the System Gadget or [n] to skip this!\033[m"
                fi
            done
            if [[ ${EINGABE} == [yY] ]]; then
                echo -n "Please wait while the System Gadget is being uninstalled..."
                echo "[${NOW} install.sh] User has chosen to uninstall the System Gadget." >> ./${LOGFILE}
                rpm -ve daccord-system-gadget >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "System Gadget uninstalled successfully\n"
                    echo "[${NOW} install.sh] daccord-system-gadget uninstalled successfully." >> ./${LOGFILE}
                    rm -fR "/opt/GuH/daccord-system-gadget" "/var/opt/GuH/daccord-system-gadget" "/opt/GuH/RuntimeEngine/webapps/system" "/opt/GuH/RuntimeEngine/webapps/system.war" >> ./${LOGFILE} 2>&1
                else
                    DEINSTERROR=1
                    echo -ne "Error while uninstalling the System Gadget\n"
                    echo "[${NOW} install.sh] Error while uninstalling the daccord-system-gadget." >> ./${LOGFILE}
                fi
            else
                echo "[${NOW} install.sh] User has chosen to skip the System Gadget." >> ./${LOGFILE}
            fi
        fi
    else
        echo -e "\nTo remove core components please remove all other components first."
    fi
    rpm -qa | grep -i '^daccord' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            if [[ ${STANDARDDEINST} == "TRUE" ]]; then
                EINGABE="y"
            else
                EINGABE=
            fi
            while [[ ${EINGABE} != [yYnN] ]]; do
                read -p $'\033[0;36mDo you want to uninstall the Runtime Engine? [y/n] \033[m' EINGABE
                if [[ ${EINGABE} != [yYnN] ]]; then
                    echo -n "\033[0;35mPlease choose between [y] to uninstall the Runtime Engine or [n] to skip this!\033[m"
                fi
            done
            if [[ ${EINGABE} == [yY] ]]; then
                echo -n "Please wait while the Runtime Engine is being uninstalled..."
                echo "[${NOW} install.sh] User has chosen to uninstall the Runtime Engine." >> ./${LOGFILE}
                rpm -ve RuntimeEngine >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "Runtime Engine uninstalled successfully\n"
                    echo "[${NOW} install.sh] RuntimeEngine uninstalled successfully." >> ./${LOGFILE}
                else
                    DEINSTERROR=1
                    echo -ne "Error while uninstalling the Runtime Engine\n"
                    echo "[${NOW} install.sh] Error while uninstalling the RuntimeEngine." >> ./${LOGFILE}
                fi
                if [[ ${DEINSTERROR} == "0" ]]; then
                    rm -fR "/opt/GuH/" "/var/opt/GuH/" >> ./${LOGFILE} 2>&1
                    rpm -qa | grep -i '^sed-[0-9]' >> /dev/null 2>&1
                    if [[ ! $? == 0 ]]; then
                        if [[ ${REDHAT} == "TRUE" ]]; then
                            yum install -y sed >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                rpm -Uvh sed* >> ./${LOGFILE} 2>&1
                                if [[ ! $? == 0 ]]; then
                                    echo "[${NOW} install.sh] Unable to install the 'sed' package." >> ./${LOGFILE}
                                fi
                            fi
                        else
                            zypper --non-interactive in sed >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                rpm -Uvh sed* >> ./${LOGFILE} 2>&1
                                if [[ ! $? == 0 ]]; then
                                    echo "[${NOW} install.sh] Unable to install the 'sed' package." >> ./${LOGFILE}
                                fi
                            fi
                        fi
                    fi
                    sed -i -e '/.*DAC_RUNTIMEENGINE_URL.*/d' -e '/.*DAC_SYSTEM_URL.*/d' -e '/.*DAC_PORTAL_PATH.*/d' /etc/environment >> ./${LOGFILE} 2>&1
                    echo -e "\nThe daccord software components have been successfully removed."
                    echo "[${NOW} install.sh] The daccord software components have been successfully removed." >> ./${LOGFILE}
                    rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
                    if [[ $? == 0 ]]; then
                        if [[ -d "/var/lib/neo4j/data/databases/daccord.db/" ]]; then
                            EINGABE=
                            while [[ ${EINGABE} != [YyNn] ]]; do
                                read -p $'\033[0;36mWould you like to completely remove the System Database as well? [y/n] \033[m' EINGABE
                                if [[ ${EINGABE} != [YyNn] ]]; then
                                    echo -e "\n\033[0;35mPlease choose between [y] to completely remove the System Database or [n] to skip!\033[m"
                                fi
                            done
                            if [[ ${EINGABE} == [Yy] ]]; then
                                DEINSTERROR=0
                                echo "[${NOW} install.sh] User has requested to remove neo4j." >> ./${LOGFILE}
                                echo -n "Please wait while the System Database is being uninstalled..."
                                systemctl stop neo4j.service >> ./${LOGFILE} 2>&1
                                systemctl disable neo4j.service >> ./${LOGFILE} 2>&1
                                rpm -ve neo4j cypher-shell >> ./${LOGFILE} 2>&1
                                if [[ $? == 0 ]]; then
                                    echo "[${NOW} install.sh] neo4j and cypher-shell uninstalled successfully." >> ./${LOGFILE}
                                else
                                    DEINSTERROR=1
                                    echo "[${NOW} install.sh] Error while uninstalling neo4j or cypher-shell." >> ./${LOGFILE}
                                fi
                                if [[ ${DEINSTERROR} == "0" ]]; then
                                    rm -fR /var/log/neo4j/ /var/lib/neo4j/ /etc/neo4j/ /usr/share/doc/neo4j/ >> ./${LOGFILE} 2>&1
                                    echo -ne "System Database uninstalled successfully\n"
                                    echo "[${NOW} install.sh] neo4j has been successfully removed." >> ./${LOGFILE}
                                else
                                    echo -ne "Error while uninstalling the System Database\n"
                                    echo -e "\033[1;31mTo troubleshoot the deinstallation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
                                fi
                            else
                                echo "[${NOW} install.sh] User has chosen to skip the System Database." >> ./${LOGFILE}
                            fi
                        fi
                    fi
                else
                    echo -e "\033[1;31mError while uninstalling some of the components.\033[m"
                    echo -e "\033[1;31mTo troubleshoot the deinstallation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
                    CANCEL
                fi
            else
                echo "[${NOW} install.sh] User has chosen to skip the Runtime Engine." >> ./${LOGFILE}
                rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    echo -n "Please wait while the Runtime Engine is being started..."
                    echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
                    systemctl daemon-reload >> ./${LOGFILE} 2>&1
                    systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        sleep 15 >> /dev/null 2>&1
                        echo -ne "Runtime Engine started successfully\n"
                        echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
                    else
                        echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                        echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                        PRESS_ANY_KEY
                    fi
                fi
                rm -f /opt/GuH/daccord_basis_install_flag >> /dev/null
            fi
        else
            if [[ ${DEINSTERROR} == "0" ]]; then
                rm -fR "/opt/GuH/" "/var/opt/GuH/" >> ./${LOGFILE} 2>&1
                rpm -qa | grep -i '^neo4j-[0-9]' >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    EINGABE=
                    while [[ ${EINGABE} != [YyNn] ]]; do
                        read -p $'\033[0;36mWould you like to completely remove the System Database as well? [y/n] \033[m' EINGABE
                        if [[ ${EINGABE} != [YyNn] ]]; then
                            echo -e "\n\033[0;35mPlease choose between [y] to completely remove the System Database or [n] to skip!\033[m"
                        fi
                    done
                    if [[ ${EINGABE} == [Yy] ]]; then
                        DEINSTERROR=0
                        echo "[${NOW} install.sh] User has requested to remove neo4j." >> ./${LOGFILE}
                        echo -n "Please wait while the System Database is being uninstalled..."
                        systemctl stop neo4j.service >> ./${LOGFILE} 2>&1
                        systemctl disable neo4j.service >> ./${LOGFILE} 2>&1
                        rpm -ve neo4j cypher-shell >> ./${LOGFILE} 2>&1
                        if [[ $? == 0 ]]; then
                            echo "[${NOW} install.sh] neo4j and cypher-shell uninstalled successfully." >> ./${LOGFILE}
                        else
                            DEINSTERROR=1
                            echo "[${NOW} install.sh] Error while uninstalling neo4j or cypher-shell." >> ./${LOGFILE}
                        fi
                        if [[ ${DEINSTERROR} == "0" ]]; then
                            rm -fR /var/log/neo4j/ /var/lib/neo4j/ /etc/neo4j/ /usr/share/doc/neo4j/ >> ./${LOGFILE} 2>&1
                            echo -ne "System Database uninstalled successfully\n"
                            echo "[${NOW} install.sh] neo4j has been successfully removed." >> ./${LOGFILE}
                        else
                            echo -ne "Error while uninstalling the System Database\n"
                            echo -e "\033[1;31mTo troubleshoot the deinstallation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
                        fi
                    fi
                fi
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    else
        if [[ ${DEINSTERROR} == "0" ]]; then
            rm -f /opt/GuH/daccord_basis_install_flag >> /dev/null
            echo -e "\nThe daccord software components have been successfully removed."
            echo "[${NOW} install.sh] The daccord software components have been successfully removed." >> ./${LOGFILE}
            rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                echo -n "Please wait while the Runtime Engine is being started..."
                echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
                systemctl daemon-reload >> ./${LOGFILE} 2>&1
                systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    sleep 15 >> /dev/null 2>&1
                    echo -ne "Runtime Engine started successfully\n"
                    echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
                else
                    echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                    echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                    PRESS_ANY_KEY
                fi
            fi
        else
            echo -e "\033[1;31mError while uninstalling some of the components.\033[m"
            echo -e "\033[1;31mTo troubleshoot the deinstallation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
            CANCEL
        fi
    fi
}

echo "[${NOW} install.sh] Script started." >> ./${LOGFILE} 2>&1
CHECK_ROOTPRIVILEGE
CHECK_OS
OUTPUT_WELCOME
CHECK_DEV
CHECK_INST_CURRENT
CHECK_PREVINSTALL_FLAG
CHECK_SERVER_TYPE
while true; do
    if [[ ${PREVINSTALL} = 1 ]]; then
        EINGABE=
        while [[ ${EINGABE} != [rRcC] ]]; do
            echo -e "[${NOW} install.sh] A previous unfinished installation has been detected on this system." >> ./${LOGFILE}
            echo -e "\nA previous unfinished installation has been detected on this system."
            read -p $'\033[0;36mWould you like to [r] remove all installed components or [c] cancel this installation? [r/c] \033[m' EINGABE
            if [[ ${EINGABE} != [rRcC] ]]; then
                echo -e "\n\033[0;35mPlease choose between [r] to remove and [c] to cancel!\033[m"
            fi
        done
        if [[ ${EINGABE} == [rR] ]]; then
            UNINSTALL
            EXIT
        else
            CANCEL
        fi
    elif [[ ${DACCORD_CURRENT_STAT} = 1 ]]; then
        EINGABE=
        while [[ ${EINGABE} != [iIuUrRcC] ]]; do
            echo -e "\nThere is at least one daccord software component already installed on this system."
            read -p $'\033[0;36mWould you like to install [i] additional components, update [u] the current component(s), remove [r] the current component(s) and cancel [c] to exit the script? [i/u/r/c] \033[m' EINGABE
            if [[ ${EINGABE} != [iIuUrRcC] ]]; then
                echo -e "\n\033[0;35mPlease choose between [i] to install additional components, [u] to update the current component(s), [r] to remove the current component(s) and [c] to cancel and exit the script!\033[m"
            fi
        done
        if [[ ${EINGABE} == [iI] ]]; then
            EINGABE=
            while [[ ${EINGABE} != [oOlL] ]]; do
                echo -e "\nThe install script provides two installation modes."
                echo -e "     [o] Online mode - The required repositories will be added and the components will be downloaded from their respective repository."
                echo -e "     [l] Local mode - The components will be installed from local rpm files. These need to be present in the directory this script is executed from."
                read -p  $'\033[0;36mPlease enter the letter for the desired installation mode and confirm with the return key: [o/l] \033[m' EINGABE
                if [[ ${EINGABE} != [oOlL] ]]; then
                    echo -e "\n\033[0;35mPlease choose between [o] for online mode and [l] for local mode!\033[m"
                fi
            done
            if [[ ${EINGABE} == [oO] ]]; then
                SET_PREVINSTALL_FLAG
                ONLINE_INSTALL
                PRESS_ANY_KEY
                exit 1
            else
                SET_PREVINSTALL_FLAG
                if ls ./*.rpm >> /dev/null 2>&1; then
                    OFFLINE_INSTALL
                    PRESS_ANY_KEY
                    exit 1
                else
                    echo -e "\nNo rpm files have been found. Please make sure you have all required rpm files in the same directory as the install script."
                    echo -e "[${NOW} install.sh] No rpm files have been found. Please make sure you have all required rpm files in the same directory as the install script." >> ./${LOGFILE}
                    CANCEL
                fi
            fi
        elif [[ ${EINGABE} == [uU] ]]; then
            EINGABE=
            while [[ ${EINGABE} != [pPcC] ]]; do
                echo -e "\033[0;31m\nAttention! Please ensure that a license with valid authorization for software updates is installed prior to performing the update. Otherwise, proper functionality of the software after the update cannot be guaranteed."
                read -p  $'\033[0;31mPlease confirm that a valid license is installed and decide if you wish to proceed [p] with the update or cancel [c] the update: [p/c] \033[m' EINGABE
                if [[ ${EINGABE} != [pPcC] ]]; then
                        echo -e "\nPlease choose between [p] to proceed with the update and [c] to cancel the update!"
                fi
            done
            if [[ ${EINGABE} == [cC] ]]; then
                EXIT
            else
                EINGABE=
                while [[ ${EINGABE} != [oOlL] ]]; do
                    echo -e "\nThe install script provides two update modes."
                    echo -e "     [o] Online mode - The components will be downloaded from their respective repositories and updated."
                    echo -e "     [l] Local mode - The components will be updated from local rpm files. These need to be present in the directory this script is executed from."
                    read -p  $'\033[0;36mPlease enter the letter for the desired update mode and confirm with the return key: [o/l] \033[m' EINGABE
                    if [[ ${EINGABE} != [oOlL] ]]; then
                        echo -e "\n\033[0;35mPlease choose between [o] for online mode and [l] for local mode!\033[m"
                    fi
                done
                if [[ ${EINGABE} == [oO] ]]; then
                    ONLINE_UPDATE
                else
                    OFFLINE_UPDATE
                fi
            fi
        elif [[ ${EINGABE} == [rR] ]]; then
            UNINSTALL
            EXIT
        else
            CANCEL
        fi
    else
        EINGABE=
        while [[ ${EINGABE} != [oOlL] ]]; do
            echo -e "\nThe install script provides two installation modes."
            echo -e "     [o] Online mode - The required repositories will be added and the components will be downloaded from their respective repository."
            echo -e "     [l] Local mode - The components will be installed from local rpm files. These need to be present in the directory this script is executed from."
            read -p  $'\033[0;36mPlease enter the letter for the desired installation mode and confirm with the return key: [o/l] \033[m' EINGABE
            if [[ ${EINGABE} != [oOlL] ]]; then
                echo -e "\n\033[0;35mPlease choose between [o] for online mode and [l] for local mode!\033[m"
            fi
        done
        if [[ ${EINGABE} == [oO] ]]; then
            SET_PREVINSTALL_FLAG
            SERVER_TYPE
            if [[ ${SERVERTYPE} == [pP] ]]; then
                if [[ ${REDHAT} == "TRUE" ]]; then
                    CHECK_NEO4J_REDHAT
                else
                    CHECK_NEO4J_SUSE
                fi
            fi
            grep "DAC_SYSTEM_URL=" /etc/environment >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                DAC_RUNTIMEENGINE_URL=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2)
                DAC_SYSTEM_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2)
                REDNSORIP=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f3 | cut -d \: -f1)
            else
                ENV_VARIABLES
            fi
            ONLINE_INSTALL
            PRESS_ANY_KEY
            exit 1
        else
            if ls ./*.rpm >> /dev/null 2>&1; then
                SET_PREVINSTALL_FLAG
                SERVER_TYPE
                if [[ ${SERVERTYPE} == [pP] ]]; then
                    CHECK_NEO4J_OFFLINE
                fi
                grep "DAC_SYSTEM_URL=" /etc/environment >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    DAC_RUNTIMEENGINE_URL=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2)
                    DAC_SYSTEM_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2)
                    REDNSORIP=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f3 | cut -d \: -f1)
                else
                    ENV_VARIABLES
                fi
                OFFLINE_INSTALL
                PRESS_ANY_KEY
                exit 1
            else
                echo -e "\nNo rpm files have been found. Please make sure you have all required rpm files in the same directory as the install script."
                echo -e "[${NOW} install.sh] No rpm files have been found. Please make sure you have all required rpm files in the same directory as the install script." >> ./${LOGFILE}
                CANCEL
            fi
        fi
    fi
done
