#!/bin/bash

##################################################################################################################
#                                                                                                                #
#                   daccord Integration Package Lifecycle v.2.2.x - installation script                          #
#                                                                                                                #
##################################################################################################################

# Set stage to either "DEVELOPMENT" or "PRODUCTION" to influence what repository the rpm files are pulled from
STAGE="PRODUCTION"
# Timestamp variable used to timestamp log entries
NOW=$(date +"%d-%m-%Y %T")
# Date variable used to give the logfile a name including the current date + hour & minute of creation
DATE=$(date +"%Y-%m-%d-%H-%M-%S")
# Logfile variable that holds the name of the logifle including the $DATE variable from above
LOGFILE=daccord-ip-lifecycle-install-${DATE}.log

# If the user id isn't 0, the script isn't being run as root and gives out a warning that it needs to be run as root to run successfully
CHECK_ROOTPRIVILEGE(){
    if [[ $EUID -ne 0 ]]; then
        echo -e "\033[0;31mThis script must be run with root privileges! Please verify that you have the needed privileges and restart the installation!\033[m"
        EXIT
    fi
}

# Check if the script is running on openSUSE or SLES or Red Hat
CHECK_OS(){
    cat /etc/os-release >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        OS=$(cat /etc/os-release | grep -e "^NAME=")
        if [[ ${OS} == 'NAME="openSUSE Leap"' ]] || [[ ${OS} == 'NAME="SLES"' ]]; then
            REDHAT="FALSE"
        else
            REDHAT="TRUE"
        fi
    else
        REDHAT="TRUE"
    fi
}

# Check if the stage variable is set to "DEVELOPMENT" and warn about it if set as such
CHECK_DEV(){
    if [[ $STAGE == "DEVELOPMENT" ]]; then
        echo -e "\033[0;31m\nWARNING! - This script is running in DEVELOPMENT mode.\033[m"
        echo -e "\033[0;31mIt will install or update from DEVELOPMENT repositories.\033[m"
    fi
}

# Creates a file to let the script notice on the next run if the previous installation didn't run through entirely
SET_PREVINSTALL_FLAG(){
    mkdir -p /opt/GuH/ >> ./${LOGFILE} 2>&1
    touch /opt/GuH/daccord_ip_lifecycle_install_flag >> ./${LOGFILE} 2>&1
}

# Check if the file that is being created by the above funtion exists on this system
CHECK_PREVINSTALL_FLAG(){
    if [[ -e /opt/GuH/daccord_ip_lifecycle_install_flag ]]; then
        PREVINSTALL=1
    fi
}

CANCEL(){
    echo "Canceling the install script!"
    echo "[${NOW} install.sh] Script has been canceled." >> ./${LOGFILE}
    exit 1
}

EXIT(){
    exit 1
}

PRESS_ANY_KEY(){
    echo -e "Press any key to continue."
    read -sn 1
}

# Check if this system already has the daccord-requests-gadget installed
CHECK_GADGET_INST(){
    DACCORD_GADGET_STAT=-1
    if [[ -e /opt/GuH/daccord-requests-gadget ]]; then
        DACCORD_GADGET_STAT=1
    fi
}

# Check if this system already has the daccord-workflow-engine installed
CHECK_ENGINE_INST(){
    DACCORD_ENGINE_STAT=-1
    if [[ -e /opt/GuH/daccord-workflow-engine ]]; then
        DACCORD_ENGINE_STAT=1
    fi
}

CHECK_NETIQ(){
    EINGABE=
    while [[ ${EINGABE} != [sSiI] ]]; do
        echo -e '\n\033[0;36mThe Workflow Engine component can be installed either as a feature-complete standalone workflow system or as an integration into a workflow system from NetIQ.\033[m'
        echo -e '\n\033[0;36mPlease note: If you want to integrate with a NetIQ workflow system, this installation must be performed on a NetIQ User Application server!\033[m'
        read -p $'\033[0;36mPlease enter [s] for installing the standalone workflow system or [i] for integration into a NetIQ system [s/i]: \033[m' EINGABE
        echo ""
        if [[ ${EINGABE} != [sSiI] ]]; then
            echo -e '\n\033[0;36mPlease choose between [s] to install the standalone workflow system or [i] to integrate into a NetIQ workflow system [s/i]: \033[m'
        fi
    done
    if [[ ${EINGABE} == [iI] ]]; then
        NETIQWE="TRUE"
    else
        NETIQWE="FALSE"
    fi
}

# Install mariadb on a SUSE system through zypper (online modus)
CHECK_MYSQL_SUSE(){
    rpm -qa | grep -i '^mariadb-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -e "\nERROR! A mariadb server is already installed on this system. Please install on a server with no mariadb server already installed!"
        echo "[${NOW} install.sh] A mariadb server is already installed on the server." >> ./${LOGFILE}
        EXIT
    else
        zypper --non-interactive in mariadb >> ./${LOGFILE} 2>&1
        while true; do
            rpm -qa | grep -i '^mariadb-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                sleep 2
                break
            fi
            grep "Failed dependencies" >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                break
            fi
            echo -n '.'
            sleep 1
        done
        systemctl start mariadb.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            rpm -qa | grep -i '^expect-[0-9]' >> /dev/null 2>&1
            if [[ $? != 0 ]]; then
                zypper --non-interactive in expect >> ./${LOGFILE} 2>&1
            fi

            MYSQLPW=
            MYSQL_ROOT_PASSWORD=camunda
            SECURE_MYSQL=$(expect -c "

            set timeout 10
            spawn mysql_secure_installation

            expect \"Enter current password for root (enter for none):\"
            send \"${MYSQLPW}\r\"

            expect \"Change the root password?\"
            send \"y\r\"

            expect \"New password:\"
            send \"${MYSQL_ROOT_PASSWORD}\r\"

            expect \"Re-enter new password:\"
            send \"${MYSQL_ROOT_PASSWORD}\r\"

            expect \"Remove anonymous users?\"
            send \"y\r\"

            expect \"Disallow root login remotely?\"
            send \"y\r\"

            expect \"Remove test database and access to it?\"
            send \"y\r\"

            expect \"Reload privilege tables now?\"
            send \"y\r\"

            expect eof
            ")

            mariadb_check=$(systemctl status mariadb.service | grep -ie "Started M")
            while [[ ${mariadb_check} = "" ]]; do
                echo -n '.'
                sleep 1
                mariadb_check=$(systemctl status mariadb.service | grep -ie "Started M")
            done

            echo "${SECURE_MYSQL}" >> /dev/null &

            MYSQL_USER=`which mysql`

            Q1="CREATE USER camunda@'localhost' IDENTIFIED BY 'camunda';"
            Q2="CREATE DATABASE camunda;"
            Q3="GRANT ALL PRIVILEGES ON *.* TO camunda@'%' IDENTIFIED BY 'camunda' WITH GRANT OPTION;"
            Q4="GRANT ALL PRIVILEGES ON *.* TO camunda@'localhost' IDENTIFIED BY 'camunda' WITH GRANT OPTION;"
            Q5="GRANT ALL PRIVILEGES ON *.* TO root@'%' IDENTIFIED BY '$MYSQL_ROOT_PASSWORD' WITH GRANT OPTION;"
            Q6="GRANT ALL PRIVILEGES ON *.* TO root@'localhost' IDENTIFIED BY '$MYSQL_ROOT_PASSWORD' WITH GRANT OPTION;"
            Q7="FLUSH PRIVILEGES;"
            SQL="${Q1}${Q2}${Q3}${Q4}${Q5}${Q6}${Q7}"
            ${MYSQL_USER} -uroot -p${MYSQL_ROOT_PASSWORD} -e "${SQL}" &

            systemctl enable mariadb.service >> ./${LOGFILE} 2>&1
        else
            echo "[${NOW} install.sh] An error occured while trying to install the mariadb server through zypper." >> ./${LOGFILE}
            echo -e "\nAn error occured while trying to install the mariadb server through zypper, please view the ${LOGFILE} file in the directory where the script file is located."
            CANCEL
        fi
    fi
}

# Install mariadb on a Red Hat system through yum (online modus)
CHECK_MYSQL_REDHAT(){
    rpm -qa | grep -i '^mariadb-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -e "\nERROR! A mariadb server is already installed on this system. Please install on a server with no mariadb server already installed!"
        echo "[${NOW} install.sh] A mariadb server is already installed on the server." >> ./${LOGFILE}
        EXIT
    else
        yum install -y mariadb-server >> ./${LOGFILE} 2>&1
        while true; do
            rpm -qa | grep -i '^mariadb-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                sleep 2
                break
            fi
            grep "Failed dependencies" >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                break
            fi
            echo -n '.'
            sleep 1
        done
        systemctl start mariadb.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            rpm -qa | grep -i '^expect-[0-9]' >> /dev/null 2>&1
            if [[ $? != 0 ]]; then
                yum install -y expect >> ./${LOGFILE} 2>&1
            fi

            MYSQLPW=
            MYSQL_ROOT_PASSWORD=camunda
            SECURE_MYSQL=$(expect -c "

            set timeout 10
            spawn mysql_secure_installation

            expect \"Enter current password for root (enter for none):\"
            send \"${MYSQLPW}\r\"

            expect \"Change the root password?\"
            send \"y\r\"

            expect \"New password:\"
            send \"${MYSQL_ROOT_PASSWORD}\r\"

            expect \"Re-enter new password:\"
            send \"${MYSQL_ROOT_PASSWORD}\r\"

            expect \"Remove anonymous users?\"
            send \"y\r\"

            expect \"Disallow root login remotely?\"
            send \"y\r\"

            expect \"Remove test database and access to it?\"
            send \"y\r\"

            expect \"Reload privilege tables now?\"
            send \"y\r\"

            expect eof
            ")

            mariadb_check=$(systemctl status mariadb.service | grep -ie "Started M")
            while [[ ${mariadb_check} = "" ]]; do
                echo -n '.'
                sleep 1
                mariadb_check=$(systemctl status mariadb.service | grep -ie "Started M")
            done

            echo "${SECURE_MYSQL}" >> /dev/null &

            MYSQL_USER=`which mysql`

            Q1="CREATE USER camunda@'localhost' IDENTIFIED BY 'camunda';"
            Q2="CREATE DATABASE camunda;"
            Q3="GRANT ALL PRIVILEGES ON *.* TO camunda@'%' IDENTIFIED BY 'camunda' WITH GRANT OPTION;"
            Q4="GRANT ALL PRIVILEGES ON *.* TO camunda@'localhost' IDENTIFIED BY 'camunda' WITH GRANT OPTION;"
            Q5="GRANT ALL PRIVILEGES ON *.* TO root@'%' IDENTIFIED BY '$MYSQL_ROOT_PASSWORD' WITH GRANT OPTION;"
            Q6="GRANT ALL PRIVILEGES ON *.* TO root@'localhost' IDENTIFIED BY '$MYSQL_ROOT_PASSWORD' WITH GRANT OPTION;"
            Q7="FLUSH PRIVILEGES;"
            SQL="${Q1}${Q2}${Q3}${Q4}${Q5}${Q6}${Q7}"
            ${MYSQL_USER} -uroot -p${MYSQL_ROOT_PASSWORD} -e "${SQL}" &

            systemctl enable mariadb.service >> ./${LOGFILE} 2>&1
        else
            echo "[${NOW} install.sh] An error occured while trying to install the mariadb server through zypper." >> ./${LOGFILE}
            echo -e "\nAn error occured while trying to install the mariadb server through zypper, please view the ${LOGFILE} file in the directory where the script file is located."
            CANCEL
        fi
    fi
}

# Install mariadb on a SUSE or Red Hat system through rpm (offline modus)
CHECK_MYSQL_LOCAL(){
    if [[ ${REDHAT} == "TRUE" ]]; then
        rpm -qa | grep -i '^mariadb-server-[0-9]' >> /dev/null 2>&1
    else
        rpm -qa | grep -i '^mariadb-[0-9]' >> /dev/null 2>&1
    fi
    if [[ $? == 0 ]]; then
        echo -e "\nERROR! A mariadb server is already installed on this system. Please install on a server with no mariadb server already installed!"
        echo "[${NOW} install.sh] A mariadb server is already installed on the server." >> ./${LOGFILE}
        EXIT
    else
        rpm -qa | grep -i '^libmariadb3-[0-9]' >> /dev/null 2>&1
        if [[ $? != 0 ]]; then
            rpm -Uvh libmariadb* >> /dev/null 2>&1
        fi
        rpm -qa | grep -i '^libopenssl1_1-[0-9]' >> /dev/null 2>&1
        if [[ $? != 0 ]]; then
            rpm -Uvh libopenssl* >> /dev/null 2>&1
        fi
        rpm -qa | grep -i '^unixODBC-[0-9]' >> /dev/null 2>&1
        if [[ $? != 0 ]]; then
            rpm -Uvh unixODBC* >> /dev/null 2>&1
        fi
        rpm -Uvh mariadb* >> ./${LOGFILE} 2>&1
        if [[ ${REDHAT} == "TRUE" ]]; then
            while true; do
                rpm -qa | grep -i '^mariadb-server-[0-9]' >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    sleep 15
                    break
                fi
                grep "Failed dependencies" >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    break
                fi
                echo -n '.'
                sleep 1
            done
        else
            while true; do
                rpm -qa | grep -i '^mariadb-[0-9]' >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    sleep 2
                    break
                fi
                grep "Failed dependencies" >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    break
                fi
                echo -n '.'
                sleep 1
            done
        fi
        systemctl start mariadb.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            rpm -qa | grep -i '^expect-[0-9]' >> /dev/null 2>&1
            if [[ $? != 0 ]]; then
                rpm -Uvh expect* >> ./${LOGFILE} 2>&1
            fi

            MYSQLPW=
            MYSQL_ROOT_PASSWORD=camunda
            SECURE_MYSQL=$(expect -c "

            set timeout 10
            spawn mysql_secure_installation

            expect \"Enter current password for root (enter for none):\"
            send \"${MYSQLPW}\r\"

            expect \"Change the root password?\"
            send \"y\r\"

            expect \"New password:\"
            send \"${MYSQL_ROOT_PASSWORD}\r\"

            expect \"Re-enter new password:\"
            send \"${MYSQL_ROOT_PASSWORD}\r\"

            expect \"Remove anonymous users?\"
            send \"y\r\"

            expect \"Disallow root login remotely?\"
            send \"y\r\"

            expect \"Remove test database and access to it?\"
            send \"y\r\"

            expect \"Reload privilege tables now?\"
            send \"y\r\"

            expect eof
            ")

            mariadb_check=$(systemctl status mariadb.service | grep -ie "Started M")
            while [[ ${mariadb_check} = "" ]]; do
                echo -n '.'
                sleep 1
                mariadb_check=$(systemctl status mariadb.service | grep -ie "Started M")
            done

            echo "${SECURE_MYSQL}" >> /dev/null &

            MYSQL_USER=`which mysql`

            Q1="CREATE USER camunda@'localhost' IDENTIFIED BY 'camunda';"
            Q2="CREATE DATABASE camunda;"
            Q3="GRANT ALL PRIVILEGES ON *.* TO camunda@'%' IDENTIFIED BY 'camunda' WITH GRANT OPTION;"
            Q4="GRANT ALL PRIVILEGES ON *.* TO camunda@'localhost' IDENTIFIED BY 'camunda' WITH GRANT OPTION;"
            Q5="GRANT ALL PRIVILEGES ON *.* TO root@'%' IDENTIFIED BY '$MYSQL_ROOT_PASSWORD' WITH GRANT OPTION;"
            Q6="GRANT ALL PRIVILEGES ON *.* TO root@'localhost' IDENTIFIED BY '$MYSQL_ROOT_PASSWORD' WITH GRANT OPTION;"
            Q7="FLUSH PRIVILEGES;"
            SQL="${Q1}${Q2}${Q3}${Q4}${Q5}${Q6}${Q7}"
            ${MYSQL_USER} -uroot -p${MYSQL_ROOT_PASSWORD} -e "${SQL}" &

            systemctl enable mariadb.service >> ./${LOGFILE} 2>&1
        else
            echo "[${NOW} install.sh] An error occured while trying to install the mariadb server through zypper." >> ./${LOGFILE}
            echo -e "\nAn error occured while trying to install the mariadb server through zypper, please view the ${LOGFILE} file in the directory where the script file is located."
            CANCEL
        fi
    fi
}

CHECK_SERVER_TYPE(){
    SERVERTYPE=""
    if [[ -e /opt/GuH/daccord-system-gadget ]]; then
        SERVERTYPE="p"
    else
        SERVERTYPE="s"
    fi
}

OUTPUT_WELCOME(){
    clear
    echo -e "\033[0;36m################################################################################\033[m"
    echo -e "\033[0;36m#                                                                              #\033[m"
    echo -e "\033[0;36m#\033[m      \033[mdaccord Integration Package Lifecycle v.2.2.x - Installation script\033[m     \033[0;36m#\033[m"
    echo -e "\033[0;36m#                                                                              #\033[m"
    echo -e "\033[0;36m################################################################################\033[m"
    echo -e " "
    echo -e "For detailed information please refer to the logfile: ${PWD}/${LOGFILE}"
}

SELFSIGNEDCERT(){
cat <<EOF> /opt/GuH/RuntimeEngine/conf/req.conf
[ req ]
default_bits              = 4096  # RSA key size
encrypt_key               = no  # Do not protect private key
default_md                = sha256  # MD to use
utf8                      = yes  # Input is UTF-8
string_mask               = utf8only  # Emit UTF-8 strings
prompt                    = no  # Do not prompt for DN
distinguished_name        = server_dn  # DN template
req_extensions            = server_reqext  # Desired extensions

[ server_dn ]
commonName                = @DNS_OR_IP@  # Should match a SAN under alt_names

[ server_reqext ]
basicConstraints          = CA:FALSE
keyUsage                  = critical,digitalSignature,keyEncipherment
extendedKeyUsage          = serverAuth
subjectKeyIdentifier      = hash
subjectAltName            = @alt_names

[alt_names]
@ENTRY@
EOF
cat <<EOF> /opt/GuH/RuntimeEngine/conf/v3.ext
subjectAltName         = @ENTRY@
EOF
    sed -i "s/@DNS_OR_IP@/${REDNSORIP}/g" /opt/GuH/RuntimeEngine/conf/req.conf
    if [[ ${DNS} == "TRUE" ]]; then
        sed -i "s/@ENTRY@/DNS.1                     = ${REDNSORIP}/g" /opt/GuH/RuntimeEngine/conf/req.conf
        sed -i "s/@ENTRY@/DNS:${REDNSORIP}/g" /opt/GuH/RuntimeEngine/conf/v3.ext
    else
        sed -i "s/@ENTRY@/IP.1                      = ${REDNSORIP}/g" /opt/GuH/RuntimeEngine/conf/req.conf
        sed -i "s/@ENTRY@/IP:${REDNSORIP}/g" /opt/GuH/RuntimeEngine/conf/v3.ext
    fi
    openssl req -new -keyout /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.key -out /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.req -batch -config /opt/GuH/RuntimeEngine/conf/req.conf >> ./${LOGFILE} 2>&1
    if [[ ${DNS} == "TRUE" ]]; then
        openssl x509 -req -in /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.req -CA /opt/GuH/RuntimeEngine/conf/PlaceholderCA.crt -CAkey /opt/GuH/RuntimeEngine/conf/PlaceholderCA.key -out /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.crt -days 3650 -sha256 -CAcreateserial -extfile /opt/GuH/RuntimeEngine/conf/v3.ext >> ./${LOGFILE} 2>&1
    else
        openssl x509 -req -in /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.req -CA /opt/GuH/RuntimeEngine/conf/PlaceholderCA.crt -CAkey /opt/GuH/RuntimeEngine/conf/PlaceholderCA.key -out /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.crt -days 3650 -sha256 -CAcreateserial -extfile /opt/GuH/RuntimeEngine/conf/v3.ext >> ./${LOGFILE} 2>&1
    fi
    cat /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.crt /opt/GuH/RuntimeEngine/conf/PlaceholderCA.crt > /opt/GuH/RuntimeEngine/conf/chain.txt
    openssl pkcs12 --password pass:changeit -export -inkey /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.key -in /opt/GuH/RuntimeEngine/conf/chain.txt -out /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.pkcs12 -name "${REDNSORIP}" >> ./${LOGFILE} 2>&1
    sed -i 's\keystoreFile="/opt/GuH/RuntimeEngine/conf/placeholder.pkcs12"\keystoreFile="/opt/GuH/RuntimeEngine/conf/'"${REDNSORIP}"'.pkcs12"\g' /opt/GuH/RuntimeEngine/conf/server.xml >> ./${LOGFILE} 2>&1
    rm /opt/GuH/RuntimeEngine/conf/chain.txt /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.key /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.crt /opt/GuH/RuntimeEngine/conf/${REDNSORIP}.req /opt/GuH/RuntimeEngine/conf/req.conf /opt/GuH/RuntimeEngine/conf/v3.ext >> /dev/null 2>&1
    chown root:RuntimeEngine /opt/GuH/RuntimeEngine/conf/* >> ./${LOGFILE} 2>&1
    chmod ug+rwx /opt/GuH/RuntimeEngine/conf/* >> ./${LOGFILE} 2>&1
}

ENV_VARIABLES(){
    REURLSCHEMA=
    while ! [[ "${REURLSCHEMA}" == [iIsS] ]]; do
        echo -e "\n\033[0;36mThe different daccord components communicate with each other. It is possibile to configure if the communication should be done securely (HTTPS and port 55443) or insecurely (HTTP and port 55080).\033[m"
        read -p $'\033[0;36mPlease choose either [i] for "Insecure" or [s] for "Secure" communication! [i/s] \033[m' REURLSCHEMA
        if [[ "${REURLSCHEMA}" == [iI] ]]; then
            export REURLSCHEMACHOSEN="http://"
            export REURLPORT=":55080"
        elif [[ "${REURLSCHEMA}" == [sS] ]]; then
            export REURLSCHEMACHOSEN="https://"
            export REURLPORT=":55443"
        fi
    done
    REDNSORIP=
    while ! [[ "${REDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z|-]*[0-9|a-zA-Z]$ ]]; do
        echo -e "\n\033[0;36mThis server needs to be accessible by the other components over the network.\033[m"
        read -p $'\033[0;36mPlease enter either the valid IP address or resolvable DNS name of this server: \033[m' REDNSORIP
        if [[ "${REDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z|-]*[0-9|a-zA-Z]$ ]]; then
            if [[ "${REDNSORIP}" =~ ^[0-9]+[.|0-9]*[0-9]$ ]]; then
                ip a | grep ${REDNSORIP} >> /dev/null 2>&1
                if [[ ! $? == 0 ]]; then
                    echo -e "\nThe IP Address that has been provided is not bound on any network adapter device on this server!"
                    REDNSORIP=
                else
                    DNS="FALSE"
                    export REDNSORIP
                fi
            else
                rpm -qa | grep -i '^bind-utils-[0-9]' >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    echo "[${NOW} install.sh] The 'bind-utils' package is already installed, proceeding with URL check." >> ./${LOGFILE}
                else
                    echo "[${NOW} install.sh] The 'bind-utils' package is not yet installed, proceeding with 'bind-utils' installation." >> ./${LOGFILE}
                    if [[ ${REDHAT} == "TRUE" ]]; then
                        yum install -y bind-utils >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh bind-utils*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'bind-utils' package." >> ./${LOGFILE}
                            fi
                        fi
                    else
                        zypper --non-interactive in bind-utils >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh bind-utils*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'bind-utils' package." >> ./${LOGFILE}
                            fi
                        fi
                    fi
                fi
                nslookup ${REDNSORIP} | grep -i "can't find" >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    echo -e "\nThe DNS Name that has been provided is not resolvable!"
                    REDNSORIP=
                else
                    DNS="TRUE"
                    export REDNSORIP
                fi
            fi
        fi
    done
    export DAC_RUNTIMEENGINE_URL="${REURLSCHEMACHOSEN}${REDNSORIP}${REURLPORT}/"
    if [[ ${SERVERTYPE} = [sS] ]]; then
        SYSURLSCHEMA=
        while ! [[ "${SYSURLSCHEMA}" == [iIsS] ]]; do
            echo -e "\n\033[0;36mThe different daccord components communicate with each other either securely (HTTPS and port 55443) or insecurely (HTTP and port 55080) depending on how the primary dacord server has been configured.\033[m"
            read -p $'\033[0;36mPlease choose either [i] for "Insecure" or [s] for "Secure" communication to the primary daccord server! [i/s] \033[m' SYSURLSCHEMA
            if [[ "${SYSURLSCHEMA}" == [iI] ]]; then
                export SYSURLSCHEMACHOSEN="http://"
                export SYSURLPORT=":55080"
            elif [[ "${SYSURLSCHEMA}" == [sS] ]]; then
                export SYSURLSCHEMACHOSEN="https://"
                export SYSURLPORT=":55443"
            fi
        done
        SYSDNSORIP=
        while ! [[ "${SYSDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z]*[0-9|a-zA-Z]$ ]]; do
            echo -e "\n\033[0;36mThis server needs to access the primary daccord server over the network.\033[m"
            read -p $'\033[0;36mPlease enter either the valid IP address or resolvable DNS name of the primary daccord server: \033[m' SYSDNSORIP
            if [[ "${SYSDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z]*[0-9|a-zA-Z]$ ]]; then
                export SYSDNSORIP
                export DAC_SYSTEM_URL="${SYSURLSCHEMACHOSEN}${SYSDNSORIP}${SYSURLPORT}/system/"
                rpm -qa | grep -i '^wget-[0-9]' >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    echo "[${NOW} install.sh] The 'wget' package is already installed, proceeding with URL check." >> ./${LOGFILE}
                else
                    echo "[${NOW} install.sh] The 'wget' package is not yet installed, proceeding with 'wget' installation." >> ./${LOGFILE}
                    if [[ ${REDHAT} == "TRUE" ]]; then
                        yum install -y wget >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh wget*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'wget' package." >> ./${LOGFILE}
                            fi
                        fi
                    else
                        zypper --non-interactive in wget >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh wget*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'wget' package." >> ./${LOGFILE}
                            fi
                        fi
                    fi
                    while true; do
                        rpm -qa | grep -i '^wget-[0-9]' >> /dev/null 2>&1
                        if [[ $? == 0 ]]; then
                            sleep 2
                            break
                        fi
                        echo -n '.'
                        sleep 1
                    done
                fi
                wget --no-check-certificate --timeout=5 --tries=1 ${DAC_SYSTEM_URL} >> ./${LOGFILE} 2>&1
                rm ./index.html >> /dev/null 2>&1
                if [[ ! $? == 0 ]]; then
                    echo -e "\nThe primary server was not reachable with the provided configuration. Please check and enter the configuration again!"
                    echo -e "Provided configuration: ${DAC_SYSTEM_URL}"
                    echo ""
                    DAC_SYSTEM_URL=""
                    SYSDNSORIP=""
                    SYSURLSCHEMA=""
                    while ! [[ "${SYSURLSCHEMA}" == [iIsS] ]]; do
                        echo -e "\n\033[0;36mThe different daccord components communicate with each other either securely (HTTPS and port 55443) or insecurely (HTTP and port 55080) depending on how the primary dacord server has been configured.\033[m"
                        read -p $'\033[0;36mPlease choose either [i] for "Insecure" or [s] for "Secure" communication to the primary daccord server! [i/s] \033[m' SYSURLSCHEMA
                        if [[ "${SYSURLSCHEMA}" == [iI] ]]; then
                            export SYSURLSCHEMACHOSEN="http://"
                            export SYSURLPORT=":55080"
                        elif [[ "${SYSURLSCHEMA}" == [sS] ]]; then
                            export SYSURLSCHEMACHOSEN="https://"
                            export SYSURLPORT=":55443"
                        fi
                    done
                fi
            fi
        done
        export DAC_SYSTEM_URL="${SYSURLSCHEMACHOSEN}${SYSDNSORIP}${SYSURLPORT}/system/"
    else
        export DAC_SYSTEM_URL="${DAC_RUNTIMEENGINE_URL}system/"
    fi
    echo "DAC_RUNTIMEENGINE_URL='${DAC_RUNTIMEENGINE_URL}'" >> /etc/environment
    echo "DAC_SYSTEM_URL='${DAC_SYSTEM_URL}'" >> /etc/environment
    source /etc/environment
}

ONLINE_INSTALL(){
    echo "[${NOW} install.sh] User has requested to start the online installation." >> ./${LOGFILE}
    EINGABE=
    while [[ ${EINGABE} != [sSaA] ]]; do
        echo -e '\n\033[0;36mTo simplify the installation of the daccord components, you can now choose between a "Simple" installation, which will install all components at once, or an "Advanced" installation, where you can choose which components to install.\033[m'
        read -p $'\033[0;36mPlease choose between [s] for the "Simple" installation or [a] for the "Advanced" installation [s/a]: \033[m' EINGABE
        echo ""
        if [[ ${EINGABE} != [sSaA] ]]; then
            echo -e '\n\033[0;36mPlease choose between [s] for the "Simple" installation or [a] for the "Advanced" installation [s/a]: \033[m'
        fi
    done
    if [[ ${EINGABE} == [sS] ]]; then
        STANDARDINST="TRUE"
    else
        STANDARDINST="FALSE"
    fi
    INSTERROR=0
    if [[ ${REDHAT} == "TRUE" ]]; then
        ONLINE_REPO_INSTALL_REDHAT
    else
        ONLINE_REPO_INSTALL_SUSE
    fi
    if [[ $INSTERROR == "0" ]]; then
        echo -e "\n\033[0;36mThe selected daccord software components have been installed successfully!\033[m"
        if [[ ! ${DACCORD_GADGET_STAT} = 1 ]]; then
            DACCORD_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f1,2,3)
            echo -e "\033[0;36mTo finalize your installation, please access the URL ${DACCORD_URL}/daccord/ with a web-browser to configure your daccord system.\033[m"
        fi
        echo -e "\n"
        rm -f /opt/GuH/daccord_ip_lifecycle_install_flag >> /dev/null
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the installation of the selected daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the installation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
    fi
}

ONLINE_REPO_INSTALL_SUSE(){
    echo "[${NOW} install.sh] Installing the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper lr daccord-ip-lifecycle-2.2-dev >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-lifecycle-2.2-dev daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        zypper lr daccord-ip-lifecycle-2.2 >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-lifecycle-2.2 daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being installed..."
        echo "[${NOW} install.sh] User has chosen to install the Runtime Engine." >> ./${LOGFILE}
        zypper --non-interactive in RuntimeEngine >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Runtime Engine installed successfully\n"
            echo "[${NOW} install.sh] RuntimeEngine installed successfully." >> ./${LOGFILE}
            SELFSIGNEDCERT
        else
            INSTERROR=1
            echo -ne "Error while installing the Runtime Engine\n"
            echo "[${NOW} install.sh] Error while installing the RuntimeEngine." >> ./${LOGFILE}
        fi
    fi
    if [[ ${SERVERTYPE} == [pP] ]]; then
        rpm -qa | grep -i '^daccord-requests-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Requests Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Requests Gadget." >> ./${LOGFILE}
            zypper --non-interactive in daccord-requests-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Requests Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-requests-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Requests Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-requests-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-workflow-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Workflow Engine component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the Workflow Engine component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            systemctl daemon-reload >> ./${LOGFILE} 2>&1
            systemctl status RuntimeEngine.service |grep -i "active (running)" >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                echo -n "Please wait while the Runtime Engine is being stopped..."
                echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
                systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    sleep 15 >> /dev/null 2>&1
                    echo -ne "The Runtime Engine has been successfully stopped\n"
                    echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
                else
                    echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
                    echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
                    PRESS_ANY_KEY
                fi
            fi
            CHECK_NETIQ
            echo -n "Please wait while the Workflow Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Workflow Engine." >> ./${LOGFILE}
            zypper --non-interactive in daccord-workflow-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Workflow Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-workflow-engine installed successfully." >> ./${LOGFILE}
                if [[ ${NETIQWE} == "TRUE" ]]; then
                    touch /opt/GuH/daccord-workflow-engine/res/netiqwe >> ./${LOGFILE} 2>&1
                    rm -Rf /opt/GuH/RuntimeEngine/conf/bpm-platform.xml /opt/GuH/RuntimeEngine/conf/server.xml_camunda /opt/GuH/RuntimeEngine/webapps/camunda /opt/GuH/RuntimeEngine/webapps/engine-rest /opt/GuH/RuntimeEngine/work/Catalina/localhost/camunda* /opt/GuH/RuntimeEngine/work/Catalina/localhost/engine-rest /opt/GuH/RuntimeEngine/lib/camunda* /opt/GuH/RuntimeEngine/lib/feel-engine-*-scala-shaded.jar /opt/GuH/RuntimeEngine/lib/freemarker-*.jar /opt/GuH/RuntimeEngine/lib/graal-sdk-*.jar /opt/GuH/RuntimeEngine/lib/groovy* /opt/GuH/RuntimeEngine/lib/h2-*.jar /opt/GuH/RuntimeEngine/lib/icu4j-*.jar /opt/GuH/RuntimeEngine/lib/java-uuid-generator-*.jar /opt/GuH/RuntimeEngine/lib/joda-time-*.jar /opt/GuH/RuntimeEngine/lib/js-*.jar /opt/GuH/RuntimeEngine/lib/mariadb-java-client-*.jar /opt/GuH/RuntimeEngine/lib/mybatis-*.jar /opt/GuH/RuntimeEngine/lib/regex-*.jar /opt/GuH/RuntimeEngine/lib/slf4j* /opt/GuH/RuntimeEngine/lib/truffle-api-*.jar >> ./${LOGFILE} 2>&1
                    systemctl daemon-reload >> ./${LOGFILE} 2>&1
                    echo -n "Please wait while the Runtime Engine is being started..."
                    echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
                    systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        sleep 15 >> /dev/null 2>&1
                        echo -ne "The Runtime Engine has been successfully started\n"
                        echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
                    else
                        echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                        echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                        PRESS_ANY_KEY
                    fi
                else
                    CHECK_MYSQL_SUSE
                    mv /opt/GuH/RuntimeEngine/conf/server.xml /opt/GuH/RuntimeEngine/conf/server.xml_org >> ./${LOGFILE} 2>&1
                    mv /opt/GuH/RuntimeEngine/conf/server.xml_camunda /opt/GuH/RuntimeEngine/conf/server.xml >> ./${LOGFILE} 2>&1
                    chown root:RuntimeEngine /opt/GuH/RuntimeEngine/conf/* >> ./${LOGFILE} 2>&1
                    chmod ug+rwx /opt/GuH/RuntimeEngine/conf/* >> ./${LOGFILE} 2>&1
                    sed -i 's\keystoreFile="/opt/GuH/RuntimeEngine/conf/placeholder.pkcs12"\keystoreFile="/opt/GuH/RuntimeEngine/conf/'"${REDNSORIP}"'.pkcs12"\g' /opt/GuH/RuntimeEngine/conf/server.xml >> ./${LOGFILE} 2>&1
                    if [[ ${SERVERTYPE} == [pP] ]]; then
                        sed -i 's/After=neo4j.service/After=neo4j.service mariadb.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                        systemctl daemon-reload >> ./${LOGFILE} 2>&1
                    else
                        sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=mariadb.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                        systemctl daemon-reload >> ./${LOGFILE} 2>&1
                    fi
                    echo -n "Please wait while the Runtime Engine is being started..."
                    echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
                    echo "" > /opt/GuH/RuntimeEngine/logs/catalina.out
                    systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        camunda_check=$(grep -Fi "Deployment of web application directory [/opt/GuH/RuntimeEngine/webapps/camunda] has finished" /opt/GuH/RuntimeEngine/logs/catalina.out)
                        while [[ ${camunda_check} = "" ]]; do
                            echo -n '.'
                            sleep 3 >> /dev/null 2>&1
                            camunda_check=$(grep -Fi "Deployment of web application directory [/opt/GuH/RuntimeEngine/webapps/camunda] has finished" /opt/GuH/RuntimeEngine/logs/catalina.out)
                        done
                        sleep 5 >> /dev/null 2>&1
                        echo -ne "The Runtime Engine has been successfully started\n"
                        echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
                    else
                        echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                        echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                        PRESS_ANY_KEY
                    fi
                    sleep 5 >> /dev/null 2>&1
                    Q1="USE camunda;"
                    Q2="INSERT INTO ACT_RU_AUTHORIZATION (ID_, REV_, TYPE_, GROUP_ID_, USER_ID_, RESOURCE_TYPE_, RESOURCE_ID_, PERMS_) VALUES ('auth-gpd', 1, 0, NULL, '*', 6, '*', 2147483647);"
                    Q3="INSERT INTO ACT_RU_AUTHORIZATION (ID_, REV_, TYPE_, GROUP_ID_, USER_ID_, RESOURCE_TYPE_, RESOURCE_ID_, PERMS_) VALUES ('auth-gpi', 1, 0, NULL, '*', 8, '*', 2147483647);"
                    SQL="${Q1}${Q2}${Q3}"
                    mysql -uroot -p${MYSQL_ROOT_PASSWORD} -e "${SQL}" &
                fi
            else
                INSTERROR=1
                echo -ne "Error while installing the Workflow Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-workflow-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Workflow Engine component." >> ./${LOGFILE}
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    systemctl status RuntimeEngine.service |grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being started..."
        echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
        systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "The Runtime Engine has been successfully started\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper mr -d daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
    else
        zypper mr -d daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
    fi
}

ONLINE_REPO_INSTALL_REDHAT(){
    echo "[${NOW} install.sh] Installing the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        if [[ ! -f "/etc/yum.repos.d/daccord-ip-lifecycle-2.2-dev.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-ip-lifecycle-2.2-dev.repo
[daccord-ip-lifecycle-2.2-dev]
name=daccord IP Lifecycle Development RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-lifecycle-2.2-dev
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        if [[ ! -f "/etc/yum.repos.d/daccord-ip-lifecycle-2.2.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-ip-lifecycle-2.2.repo
[daccord-ip-lifecycle-2.2]
name=daccord IP Lifecycle RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-lifecycle-2.2
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being installed..."
        echo "[${NOW} install.sh] User has chosen to install the Runtime Engine." >> ./${LOGFILE}
        yum install -y RuntimeEngine >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Runtime Engine installed successfully\n"
            echo "[${NOW} install.sh] RuntimeEngine installed successfully." >> ./${LOGFILE}
            SELFSIGNEDCERT
        else
            INSTERROR=1
            echo -ne "Error while installing the Runtime Engine\n"
            echo "[${NOW} install.sh] Error while installing the RuntimeEngine." >> ./${LOGFILE}
        fi
    fi
    if [[ ${SERVERTYPE} == [pP] ]]; then
        rpm -qa | grep -i '^daccord-requests-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Requests Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Requests Gadget." >> ./${LOGFILE}
            yum install -y daccord-requests-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Requests Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-requests-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Requests Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-requests-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-workflow-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Workflow Engine component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the Workflow Engine component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            systemctl daemon-reload >> ./${LOGFILE} 2>&1
            systemctl status RuntimeEngine.service |grep -i "active (running)" >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                echo -n "Please wait while the Runtime Engine is being stopped..."
                echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
                systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    sleep 15 >> /dev/null 2>&1
                    echo -ne "The Runtime Engine has been successfully stopped\n"
                    echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
                else
                    echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
                    echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
                    PRESS_ANY_KEY
                fi
            fi
            CHECK_NETIQ
            echo -n "Please wait while the Workflow Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Workflow Engine." >> ./${LOGFILE}
            yum install -y daccord-workflow-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Workflow Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-workflow-engine installed successfully." >> ./${LOGFILE}
                if [[ ${NETIQWE} == "TRUE" ]]; then
                    touch /opt/GuH/daccord-workflow-engine/res/netiqwe >> ./${LOGFILE} 2>&1
                    rm -Rf /opt/GuH/RuntimeEngine/conf/bpm-platform.xml /opt/GuH/RuntimeEngine/conf/server.xml_camunda /opt/GuH/RuntimeEngine/webapps/camunda /opt/GuH/RuntimeEngine/webapps/engine-rest /opt/GuH/RuntimeEngine/work/Catalina/localhost/camunda* /opt/GuH/RuntimeEngine/work/Catalina/localhost/engine-rest /opt/GuH/RuntimeEngine/lib/camunda* /opt/GuH/RuntimeEngine/lib/feel-engine-*-scala-shaded.jar /opt/GuH/RuntimeEngine/lib/freemarker-*.jar /opt/GuH/RuntimeEngine/lib/graal-sdk-*.jar /opt/GuH/RuntimeEngine/lib/groovy* /opt/GuH/RuntimeEngine/lib/h2-*.jar /opt/GuH/RuntimeEngine/lib/icu4j-*.jar /opt/GuH/RuntimeEngine/lib/java-uuid-generator-*.jar /opt/GuH/RuntimeEngine/lib/joda-time-*.jar /opt/GuH/RuntimeEngine/lib/js-*.jar /opt/GuH/RuntimeEngine/lib/mariadb-java-client-*.jar /opt/GuH/RuntimeEngine/lib/mybatis-*.jar /opt/GuH/RuntimeEngine/lib/regex-*.jar /opt/GuH/RuntimeEngine/lib/slf4j* /opt/GuH/RuntimeEngine/lib/truffle-api-*.jar >> ./${LOGFILE} 2>&1
                    systemctl daemon-reload >> ./${LOGFILE} 2>&1
                    echo -n "Please wait while the Runtime Engine is being started..."
                    echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
                    systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        sleep 15 >> /dev/null 2>&1
                        echo -ne "The Runtime Engine has been successfully started\n"
                        echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
                    else
                        echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                        echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                        PRESS_ANY_KEY
                    fi
                else
                    CHECK_MYSQL_REDHAT
                    mv /opt/GuH/RuntimeEngine/conf/server.xml /opt/GuH/RuntimeEngine/conf/server.xml_org >> ./${LOGFILE} 2>&1
                    mv /opt/GuH/RuntimeEngine/conf/server.xml_camunda /opt/GuH/RuntimeEngine/conf/server.xml >> ./${LOGFILE} 2>&1
                    chown root:RuntimeEngine /opt/GuH/RuntimeEngine/conf/* >> ./${LOGFILE} 2>&1
                    chmod ug+rwx /opt/GuH/RuntimeEngine/conf/* >> ./${LOGFILE} 2>&1
                    sed -i 's\keystoreFile="/opt/GuH/RuntimeEngine/conf/placeholder.pkcs12"\keystoreFile="/opt/GuH/RuntimeEngine/conf/'"${REDNSORIP}"'.pkcs12"\g' /opt/GuH/RuntimeEngine/conf/server.xml >> ./${LOGFILE} 2>&1
                    if [[ ${SERVERTYPE} == [pP] ]]; then
                        sed -i 's/After=neo4j.service/After=neo4j.service mariadb.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                        systemctl daemon-reload >> ./${LOGFILE} 2>&1
                    else
                        sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=mariadb.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                        systemctl daemon-reload >> ./${LOGFILE} 2>&1
                    fi
                    echo -n "Please wait while the Runtime Engine is being started..."
                    echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
                    echo "" > /opt/GuH/RuntimeEngine/logs/catalina.out
                    systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        camunda_check=$(grep -Fi "Deployment of web application directory [/opt/GuH/RuntimeEngine/webapps/camunda] has finished" /opt/GuH/RuntimeEngine/logs/catalina.out)
                        while [[ ${camunda_check} = "" ]]; do
                            echo -n '.'
                            sleep 3 >> /dev/null 2>&1
                            camunda_check=$(grep -Fi "Deployment of web application directory [/opt/GuH/RuntimeEngine/webapps/camunda] has finished" /opt/GuH/RuntimeEngine/logs/catalina.out)
                        done
                        sleep 5 >> /dev/null 2>&1
                        echo -ne "The Runtime Engine has been successfully started\n"
                        echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
                    else
                        echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                        echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                        PRESS_ANY_KEY
                    fi
                    sleep 5 >> /dev/null 2>&1
                    Q1="USE camunda;"
                    Q2="INSERT INTO ACT_RU_AUTHORIZATION (ID_, REV_, TYPE_, GROUP_ID_, USER_ID_, RESOURCE_TYPE_, RESOURCE_ID_, PERMS_) VALUES ('auth-gpd', 1, 0, NULL, '*', 6, '*', 2147483647);"
                    Q3="INSERT INTO ACT_RU_AUTHORIZATION (ID_, REV_, TYPE_, GROUP_ID_, USER_ID_, RESOURCE_TYPE_, RESOURCE_ID_, PERMS_) VALUES ('auth-gpi', 1, 0, NULL, '*', 8, '*', 2147483647);"
                    SQL="${Q1}${Q2}${Q3}"
                    mysql -uroot -p${MYSQL_ROOT_PASSWORD} -e "${SQL}" &
                fi
            else
                INSTERROR=1
                echo -ne "Error while installing the Workflow Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-workflow-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Workflow Engine component." >> ./${LOGFILE}
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    systemctl status RuntimeEngine.service |grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being started..."
        echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
        systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "The Runtime Engine has been successfully started\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        yum config-manager --disable daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
    else
        yum config-manager --disable daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
    fi
}

OFFLINE_INSTALL(){
    echo "[${NOW} install.sh] User has requested to start the offline installation." >> ./${LOGFILE}
    EINGABE=
    while [[ ${EINGABE} != [sSaA] ]]; do
        echo -e '\n\033[0;36mTo simplify the installation of the daccord components, you can now choose between a "Simple" installation, which will install all components at once, or an "Advanced" installation, where you can choose which components to install.\033[m'
        echo ""
        read -p $'\033[0;36mPlease choose between [s] for the "Simple" installation or [a] for the "Advanced" installation [s/a]: \033[m' EINGABE
        if [[ ${EINGABE} != [sSaA] ]]; then
            echo -e '\n\033[0;36mPlease choose between [s] for the "Simple" installation or [a] for the "Advanced" installation [s/a]: \033[m'
        fi
    done
    if [[ ${EINGABE} == [sS] ]]; then
        STANDARDINST="TRUE"
    else
        STANDARDINST="FALSE"
    fi
    INSTERROR=0
    OFFLINE_RPM_INSTALL
    if [[ $INSTERROR == "0" ]]; then
        echo -e "\nThe selected daccord software components have been installed successfully!"
        if [[ ! ${DACCORD_CURRENT_STAT} = 1 ]]; then
            DACCORD_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f1,2,3)
            echo -e "\033[0;36mTo finalize your installation, please access the URL ${DACCORD_URL}/daccord/ with a web-browser to configure your daccord system.\033[m"
        fi
        echo -e "\n"
        rm -f /opt/GuH/daccord_ip_lifecycle_install_flag >> /dev/null
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the installation of the selected daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the installation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
    fi
}

OFFLINE_RPM_INSTALL(){
    echo "[${NOW} install.sh] Installing the daccord software components from local rpm files" >> ./${LOGFILE}
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being installed..."
        echo "[${NOW} install.sh] User has chosen to install the Runtime Engine." >> ./${LOGFILE}
        rpm -Uvh RuntimeEngine* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Runtime Engine installed successfully\n"
            echo "[${NOW} install.sh] RuntimeEngine installed successfully." >> ./${LOGFILE}
            SELFSIGNEDCERT
        else
            INSTERROR=1
            echo -ne "Error while installing the Runtime Engine\n"
            echo "[${NOW} install.sh] Error while installing the RuntimeEngine." >> ./${LOGFILE}
        fi
    fi
    if [[ ${SERVERTYPE} == [pP] ]]; then
        rpm -qa | grep -i '^daccord-requests-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Requests Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Requests Gadget." >> ./${LOGFILE}
            rpm -Uvh daccord-requests-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Requests Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-requests-gadge installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Requests Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-requests-gadge." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-workflow-engine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        if [[ ${STANDARDINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to install the Workflow Engine component? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -e "\n\033[0;35mPlease choose between [y] to install the Workflow Engine component or [n] to skip this component!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            systemctl daemon-reload >> ./${LOGFILE} 2>&1
            systemctl status RuntimeEngine.service |grep -i "active (running)" >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                echo -n "Please wait while the Runtime Engine is being stopped..."
                echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
                systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    sleep 15 >> /dev/null 2>&1
                    echo -ne "The Runtime Engine has been successfully stopped\n"
                    echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
                else
                    echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
                    echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
                    PRESS_ANY_KEY
                fi
            fi
            CHECK_NETIQ
            echo -n "Please wait while the Workflow Engine is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Workflow Engine." >> ./${LOGFILE}
            rpm -Uvh daccord-workflow-engine* >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Workflow Engine installed successfully\n"
                echo "[${NOW} install.sh] daccord-workflow-engine installed successfully." >> ./${LOGFILE}
                if [[ ${NETIQWE} == "TRUE" ]]; then
                    touch /opt/GuH/daccord-workflow-engine/res/netiqwe >> ./${LOGFILE} 2>&1
                    rm -Rf /opt/GuH/RuntimeEngine/conf/bpm-platform.xml /opt/GuH/RuntimeEngine/conf/server.xml_camunda /opt/GuH/RuntimeEngine/webapps/camunda /opt/GuH/RuntimeEngine/webapps/engine-rest /opt/GuH/RuntimeEngine/work/Catalina/localhost/camunda* /opt/GuH/RuntimeEngine/work/Catalina/localhost/engine-rest /opt/GuH/RuntimeEngine/lib/camunda* /opt/GuH/RuntimeEngine/lib/feel-engine-*-scala-shaded.jar /opt/GuH/RuntimeEngine/lib/freemarker-*.jar /opt/GuH/RuntimeEngine/lib/graal-sdk-*.jar /opt/GuH/RuntimeEngine/lib/groovy* /opt/GuH/RuntimeEngine/lib/h2-*.jar /opt/GuH/RuntimeEngine/lib/icu4j-*.jar /opt/GuH/RuntimeEngine/lib/java-uuid-generator-*.jar /opt/GuH/RuntimeEngine/lib/joda-time-*.jar /opt/GuH/RuntimeEngine/lib/js-*.jar /opt/GuH/RuntimeEngine/lib/mariadb-java-client-*.jar /opt/GuH/RuntimeEngine/lib/mybatis-*.jar /opt/GuH/RuntimeEngine/lib/regex-*.jar /opt/GuH/RuntimeEngine/lib/slf4j* /opt/GuH/RuntimeEngine/lib/truffle-api-*.jar >> ./${LOGFILE} 2>&1
                    systemctl daemon-reload >> ./${LOGFILE} 2>&1
                    echo -n "Please wait while the Runtime Engine is being started..."
                    echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
                    systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        sleep 15 >> /dev/null 2>&1
                        echo -ne "The Runtime Engine has been successfully started\n"
                        echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
                    else
                        echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                        echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                        PRESS_ANY_KEY
                    fi
                else
                    CHECK_MYSQL_LOCAL
                    mv /opt/GuH/RuntimeEngine/conf/server.xml /opt/GuH/RuntimeEngine/conf/server.xml_org >> ./${LOGFILE} 2>&1
                    mv /opt/GuH/RuntimeEngine/conf/server.xml_camunda /opt/GuH/RuntimeEngine/conf/server.xml >> ./${LOGFILE} 2>&1
                    chown root:RuntimeEngine /opt/GuH/RuntimeEngine/conf/* >> ./${LOGFILE} 2>&1
                    chmod ug+rwx /opt/GuH/RuntimeEngine/conf/* >> ./${LOGFILE} 2>&1
                    sed -i 's\keystoreFile="/opt/GuH/RuntimeEngine/conf/placeholder.pkcs12"\keystoreFile="/opt/GuH/RuntimeEngine/conf/'"${REDNSORIP}"'.pkcs12"\g' /opt/GuH/RuntimeEngine/conf/server.xml >> ./${LOGFILE} 2>&1
                    if [[ ${SERVERTYPE} == [pP] ]]; then
                        sed -i 's/After=neo4j.service/After=neo4j.service mariadb.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                        systemctl daemon-reload >> ./${LOGFILE} 2>&1
                    else
                        sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=mariadb.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                        systemctl daemon-reload >> ./${LOGFILE} 2>&1
                    fi
                    echo -n "Please wait while the Runtime Engine is being started..."
                    echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
                    echo "" > /opt/GuH/RuntimeEngine/logs/catalina.out
                    systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        camunda_check=$(grep -Fi "Deployment of web application directory [/opt/GuH/RuntimeEngine/webapps/camunda] has finished" /opt/GuH/RuntimeEngine/logs/catalina.out)
                        while [[ ${camunda_check} = "" ]]; do
                            echo -n '.'
                            sleep 3 >> /dev/null 2>&1
                            camunda_check=$(grep -Fi "Deployment of web application directory [/opt/GuH/RuntimeEngine/webapps/camunda] has finished" /opt/GuH/RuntimeEngine/logs/catalina.out)
                        done
                        sleep 5 >> /dev/null 2>&1
                        echo -ne "The Runtime Engine has been successfully started\n"
                        echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
                    else
                        echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                        echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                        PRESS_ANY_KEY
                    fi
                    sleep 5 >> /dev/null 2>&1
                    Q1="USE camunda;"
                    Q2="INSERT INTO ACT_RU_AUTHORIZATION (ID_, REV_, TYPE_, GROUP_ID_, USER_ID_, RESOURCE_TYPE_, RESOURCE_ID_, PERMS_) VALUES ('auth-gpd', 1, 0, NULL, '*', 6, '*', 2147483647);"
                    Q3="INSERT INTO ACT_RU_AUTHORIZATION (ID_, REV_, TYPE_, GROUP_ID_, USER_ID_, RESOURCE_TYPE_, RESOURCE_ID_, PERMS_) VALUES ('auth-gpi', 1, 0, NULL, '*', 8, '*', 2147483647);"
                    SQL="${Q1}${Q2}${Q3}"
                    mysql -uroot -p${MYSQL_ROOT_PASSWORD} -e "${SQL}" &
                fi
            else
                INSTERROR=1
                echo -ne "Error while installing the Workflow Engine\n"
                echo "[${NOW} install.sh] Error while installing the daccord-workflow-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Workflow Engine component." >> ./${LOGFILE}
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    systemctl status RuntimeEngine.service |grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being started..."
        echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
        systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "The Runtime Engine has been successfully started\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
}

ONLINE_UPDATE(){
    echo "[${NOW} install.sh] User has requested to start the online update." >> ./${LOGFILE}
    UPERROR=0
    echo ""
    if [[ ${REDHAT} == "TRUE" ]]; then
        ONLINE_REPO_UPDATE_REDHAT
    else
        ONLINE_REPO_UPDATE_SUSE
    fi
    if [[ $UPERROR == "0" ]]; then
        echo -e "\nThe daccord software components have been updated successfully!"
        DACCORD_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f1,2,3)
        echo -e "\033[0;36mTo finalize your installation, please access the URL ${DACCORD_URL}/daccord/ with a web-browser to configure your daccord system.\033[m"
        echo -e "\n"
        EXIT
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the update of the daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the update, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
        EXIT
    fi
}

ONLINE_REPO_UPDATE_SUSE(){
    echo "[${NOW} install.sh] Updating the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper lr daccord-ip-lifecycle-2.2-dev >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-lifecycle-2.2-dev daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        zypper lr daccord-ip-lifecycle-2.2 >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-lifecycle-2.2 daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -ne "ERROR! No Runtime Engine found on the system! Cancelling the update!"
        echo "[${NOW} install.sh] ERROR! No RuntimeEngine found on the system! Cancelling the update!" >> ./${LOGFILE}
        CANCEL
    fi
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        REMANUALSTOP="TRUE"
        echo "[${NOW} install.sh] The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!" >> ./${LOGFILE}
        echo -e "The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!\n"
    else
        REMANUALSTOP="FALSE"
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for component updates..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    if [[ ${SERVERTYPE} != [pP] ]]; then
        zypper --non-interactive up -D RuntimeEngine | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Runtime Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the RuntimeEngine." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Runtime Engine is being updated..."
            zypper --non-interactive up RuntimeEngine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Runtime Engine updated successfully\n"
                echo "[${NOW} install.sh] RuntimeEngine updated successfully." >> ./${LOGFILE}
                sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=mariadb.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                systemctl daemon-reload >> ./${LOGFILE} 2>&1
            else
                UPERROR=1
                echo -ne "Error while updating the Runtime Engine\n"
                echo "[${NOW} install.sh] Error while updating the RuntimeEngine." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-requests-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-requests-gadget | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Requests Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-requests-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Requests Gadget is being updated..."
            zypper --non-interactive up daccord-requests-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Requests Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-requests-gadget updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Requests Gadget\n"
                echo "[${NOW} install.sh] Error while updating the requests-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-workflow-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-workflow-engine | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Workflow Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-workflow-engine." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Workflow Engine is being updated..."
            zypper --non-interactive up daccord-workflow-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Workflow Engine updated successfully\n"
                echo "[${NOW} install.sh] daccord-workflow-engine updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Workflow Engine\n"
                echo "[${NOW} install.sh] Error while updating the daccord-workflow-engine." >> ./${LOGFILE}
            fi
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    if [[ ${REMANUALSTOP} == "FALSE" ]]; then
        systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Runtime Engine is being started..."
            echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
            systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 15 >> /dev/null 2>&1
                echo -ne "The Runtime Engine has been successfully started\n"
                echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper mr -d daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
    else
        zypper mr -d daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
    fi
}

ONLINE_REPO_UPDATE_REDHAT(){
    echo "[${NOW} install.sh] Updating the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        if [[ ! -f "/etc/yum.repos.d/daccord-ip-lifecycle-2.2-dev.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-ip-lifecycle-2.2-dev.repo
[daccord-ip-lifecycle-2.2-dev]
name=daccord IP Lifecycle Development RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-lifecycle-2.2-dev
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        if [[ ! -f "/etc/yum.repos.d/daccord-ip-lifecycle-2.2.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-ip-lifecycle-2.2.repo
[daccord-ip-lifecycle-2.2]
name=daccord IP Lifecycle RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-lifecycle-2.2
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    yum clean all >> /dev/null 2>&1
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -ne "ERROR! No Runtime Engine found on the system! Cancelling the update!"
        echo "[${NOW} install.sh] ERROR! No RuntimeEngine found on the system! Cancelling the update!" >> ./${LOGFILE}
        CANCEL
    fi
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        REMANUALSTOP="TRUE"
        echo "[${NOW} install.sh] The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!" >> ./${LOGFILE}
        echo -e "The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!\n"
    else
        REMANUALSTOP="FALSE"
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for component updates..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    if [[ ${SERVERTYPE} != [pP] ]]; then
        yum check-update RuntimeEngine | grep daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Runtime Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the RuntimeEngine." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Runtime Engine is being updated..."
            yum upgrade -y RuntimeEngine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Runtime Engine updated successfully\n"
                echo "[${NOW} install.sh] RuntimeEngine updated successfully." >> ./${LOGFILE}
                sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=mariadb.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                systemctl daemon-reload >> ./${LOGFILE} 2>&1
            else
                UPERROR=1
                echo -ne "Error while updating the Runtime Engine\n"
                echo "[${NOW} install.sh] Error while updating the RuntimeEngine." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-requests-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-requests-gadget | grep daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Requests Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-requests-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Requests Gadget is being updated..."
            yum upgrade -y daccord-requests-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Requests Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-requests-gadget updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Requests Gadget\n"
                echo "[${NOW} install.sh] Error while updating the requests-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-workflow-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-workflow-engine | grep daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Workflow Engine\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-workflow-engine." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Workflow Engine is being updated..."
            yum upgrade -y daccord-workflow-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Workflow Engine updated successfully\n"
                echo "[${NOW} install.sh] daccord-workflow-engine updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Workflow Engine\n"
                echo "[${NOW} install.sh] Error while updating the daccord-workflow-engine." >> ./${LOGFILE}
            fi
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    if [[ ${REMANUALSTOP} == "FALSE" ]]; then
        systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Runtime Engine is being started..."
            echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
            systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 15 >> /dev/null 2>&1
                echo -ne "The Runtime Engine has been successfully started\n"
                echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        yum config-manager --disable daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
    else
        yum config-manager --disable daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
    fi
}

OFFLINE_UPDATE(){
    echo "[${NOW} install.sh] User has requested to start the offline update." >> ./${LOGFILE}
    UPERROR=0
    echo ""
    OFFLINE_RPM_UPDATE
    if [[ $UPERROR == "0" ]]; then
        echo -e "\nThe daccord software components have been updated successfully!"
        DACCORD_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f1,2,3)
        echo -e "\033[0;36mTo finalize your installation, please access the URL ${DACCORD_URL}/daccord/ with a web-browser to configure your daccord system.\033[m"
        echo -e "\n"
        EXIT
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the update of the daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the update, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
        EXIT
    fi
}

OFFLINE_RPM_UPDATE(){
    echo "[${NOW} install.sh] Updating the daccord software components from local rpm files." >> ./${LOGFILE}
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -ne "ERROR! No Runtime Engine found on the system! Cancelling the update!"
        echo "[${NOW} install.sh] ERROR! No RuntimeEngine found on the system! Cancelling the update!" >> ./${LOGFILE}
        CANCEL
    fi
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        REMANUALSTOP="TRUE"
        echo "[${NOW} install.sh] The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!" >> ./${LOGFILE}
        echo -e "The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!\n"
    else
        REMANUALSTOP="FALSE"
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for component updates..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    if [[ ${SERVERTYPE} != [pP] ]]; then
        rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            echo -n "Please wait while the Runtime Engine is being updated..."
            rpm -Uvh RuntimeEngine* >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Runtime Engine updated successfully\n"
                echo "[${NOW} install.sh] RuntimeEngine updated successfully." >> ./${LOGFILE}
                sed -i 's/Before=multi-user.target graphical.target/Before=multi-user.target graphical.target\nAfter=mariadb.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                systemctl daemon-reload >> ./${LOGFILE} 2>&1
            else
                UPERROR=1
                echo -ne "Error while updating the Runtime Engine\n"
                echo "[${NOW} install.sh] Error while updating the RuntimeEngine." >> ./${LOGFILE}
            fi
        fi
    fi
    rpm -qa | grep -i '^daccord-requests-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Requests Gadget is being updated..."
        rpm -Uvh daccord-requests-gadget* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Requests Gadget updated successfully\n"
            echo "[${NOW} install.sh] daccord-requests-gadget updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the Requests Gadget\n"
            echo "[${NOW} install.sh] Error while updating the requests-gadget." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-workflow-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Workflow Engine is being updated..."
        rpm -Uvh daccord-workflow-engine* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Workflow Engine updated successfully\n"
            echo "[${NOW} install.sh] daccord-workflow-engine updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the Workflow Engine\n"
            echo "[${NOW} install.sh] Error while updating the daccord-workflow-engine." >> ./${LOGFILE}
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    if [[ ${REMANUALSTOP} == "FALSE" ]]; then
        systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Runtime Engine is being started..."
            echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
            systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 15 >> /dev/null 2>&1
                echo -ne "The Runtime Engine has been successfully started\n"
                echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
}

UNINSTALL(){
    DEINSTERROR=0
    echo "[${NOW} install.sh] User has requested to start the deinstallation." >> ./${LOGFILE}
    EINGABE=
    while [[ ${EINGABE} != [sSaA] ]]; do
        echo -e '\n\033[0;36mTo simplify the deinstallation of the daccord components, you can now choose between a "Simple" deinstallation, which will uninstall all components at once, or an "Advanced" deinstallation, where you can choose which components to uninstall.\033[m'
        read -p $'\033[0;36mPlease choose between [s] for the "Simple" deinstallation or [a] for the "Advanced" deinstallation [s/a]: \033[m' EINGABE
        echo ""
        if [[ ${EINGABE} != [sSaA] ]]; then
            echo -e '\n\033[0;36mPlease choose between [s] for the "Simple" deinstallation or [a] for the "Advanced" deinstallation [s/a]: \033[m'
        fi
    done
    if [[ ${EINGABE} == [sS] ]]; then
        STANDARDDEINST="TRUE"
    else
        STANDARDDEINST="FALSE"
    fi
    if [[ ${REDHAT} == "TRUE" ]]; then
        if [[ -f "/etc/yum.repos.d/daccord-ip-lifecycle-2.2-dev.repo" ]]; then
            rm -f /etc/yum.repos.d/daccord-ip-lifecycle-2.2-dev.repo >> ./${LOGFILE} 2>&1
        fi
        if [[ -f "/etc/yum.repos.d/daccord-ip-lifecycle-2.2.repo" ]]; then
            rm -f /etc/yum.repos.d/daccord-ip-lifecycle-2.2.repo >> ./${LOGFILE} 2>&1
        fi
    else
        zypper lr daccord-ip-lifecycle-2.2-dev >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper rr daccord-ip-lifecycle-2.2-dev >> ./${LOGFILE} 2>&1
        fi
        zypper lr daccord-ip-lifecycle-2.2 >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper rr daccord-ip-lifecycle-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for component deinstallations..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    rpm -qa | grep -i '^daccord-workflow-engine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        if [[ ${STANDARDDEINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to uninstall the Workflow Engine? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to uninstall the Workflow Engine or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            if [[ ! -e "/opt/GuH/daccord-workflow-engine/res/netiqwe" ]]; then
                systemctl stop mariadb.service >> ./${LOGFILE} 2>&1
                systemctl disable mariadb.service >> ./${LOGFILE} 2>&1
                if [[ ${REDHAT} == "TRUE" ]]; then
                    rpm -ve mariadb-server >> ./${LOGFILE} 2>&1
                    if [[ ! $? == 0 ]]; then
                        echo "[${NOW} install.sh] Error while uninstalling mariadb." >> ./${LOGFILE}
                    fi
                else
                    rpm -ve mariadb >> ./${LOGFILE} 2>&1
                    if [[ ! $? == 0 ]]; then
                        echo "[${NOW} install.sh] Error while uninstalling mariadb." >> ./${LOGFILE}
                    fi
                fi
                rpm -ve mariadb-client >> ./${LOGFILE} 2>&1
                if [[ ! $? == 0 ]]; then
                    echo "[${NOW} install.sh] Error while uninstalling mariadb-client." >> ./${LOGFILE}
                fi
                rpm -ve mariadb-errormessages >> ./${LOGFILE} 2>&1
                if [[ ! $? == 0 ]]; then
                    echo "[${NOW} install.sh] Error while uninstalling mariadb-errormessages." >> ./${LOGFILE}
                fi
                rm -fR /var/log/mysql/ /var/lib/mysql /run/mysql/ /usr/share/mariadb/ >> ./${LOGFILE} 2>&1
                if [[ ${SERVERTYPE} == [pP] ]]; then
                    sed -i 's/After=neo4j.service mariadb.service/After=neo4j.service/g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                    systemctl daemon-reload >> ./${LOGFILE} 2>&1
                else
                    sed -i 's/After=mariadb.service//g' /usr/lib/systemd/system/RuntimeEngine.service >> ./${LOGFILE} 2>&1
                    systemctl daemon-reload >> ./${LOGFILE} 2>&1
                fi
            fi
            echo -n "Please wait while the Workflow Engine is being uninstalled..."
            echo "[${NOW} install.sh] User has chosen to uninstall the Workflow Engine." >> ./${LOGFILE}
            rpm -ve daccord-workflow-engine >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Workflow Engine uninstalled successfully\n"
                echo "[${NOW} install.sh] daccord-workflow-engine uninstalled successfully." >> ./${LOGFILE}
                rm -fR "/opt/GuH/daccord-workflow-engine" "/var/opt/GuH/daccord-workflow-engine" "/opt/GuH/RuntimeEngine/webapps/workflowengine" "/opt/GuH/RuntimeEngine/webapps/workflowengine.war" "/opt/GuH/RuntimeEngine/conf/server.xml" >> ./${LOGFILE} 2>&1
                mv "/opt/GuH/RuntimeEngine/conf/server.xml_org" "/opt/GuH/RuntimeEngine/conf/server.xml"
            else
                DEINSTERROR=1
                echo -ne "Error while uninstalling the Workflow Engine\n"
                echo "[${NOW} install.sh] Error while uninstalling the daccord-workflow-engine." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Workflow Engine." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord-requests-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        if [[ ${STANDARDDEINST} == "TRUE" ]]; then
            EINGABE="y"
        else
            EINGABE=
        fi
        while [[ ${EINGABE} != [yYnN] ]]; do
            read -p $'\033[0;36mDo you want to uninstall the Requests Gadget? [y/n] \033[m' EINGABE
            if [[ ${EINGABE} != [yYnN] ]]; then
                echo -n "\033[0;35mPlease choose between [y] to uninstall the Requests Gadget or [n] to skip this!\033[m"
            fi
        done
        if [[ ${EINGABE} == [yY] ]]; then
            echo -n "Please wait while the Requests Gadget is being uninstalled..."
            echo "[${NOW} install.sh] User has chosen to uninstall the Requests Gadget." >> ./${LOGFILE}
            rpm -ve daccord-requests-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Requests Gadget uninstalled successfully\n"
                echo "[${NOW} install.sh] daccord-requests-gadget uninstalled successfully." >> ./${LOGFILE}
                rm -fR "/opt/GuH/daccord-requests-gadget" "/var/opt/GuH/daccord-requests-gadget" "/opt/GuH/RuntimeEngine/webapps/requests" "/opt/GuH/RuntimeEngine/webapps/requests.war" >> ./${LOGFILE} 2>&1
            else
                DEINSTERROR=1
                echo -ne "Error while uninstalling the Requests Gadget\n"
                echo "[${NOW} install.sh] Error while uninstalling the daccord-requests-gadget." >> ./${LOGFILE}
            fi
        else
            echo "[${NOW} install.sh] User has chosen to skip the Requests Gadget." >> ./${LOGFILE}
        fi
    fi
    rpm -qa | grep -i '^daccord' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            if [[ ${STANDARDDEINST} == "TRUE" ]]; then
                EINGABE="y"
            else
                EINGABE=
            fi
            while [[ ${EINGABE} != [yYnN] ]]; do
                read -p $'\033[0;36mDo you want to uninstall the Runtime Engine? [y/n] \033[m' EINGABE
                if [[ ${EINGABE} != [yYnN] ]]; then
                    echo -n "\033[0;35mPlease choose between [y] to uninstall the Runtime Engine or [n] to skip this!\033[m"
                fi
            done
            if [[ ${EINGABE} == [yY] ]]; then
                echo -n "Please wait while the Runtime Engine is being uninstalled..."
                echo "[${NOW} install.sh] User has chosen to uninstall the Runtime Engine." >> ./${LOGFILE}
                rpm -ve RuntimeEngine >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    echo -ne "Runtime Engine uninstalled successfully\n"
                    echo "[${NOW} install.sh] RuntimeEngine uninstalled successfully." >> ./${LOGFILE}
                else
                    DEINSTERROR=1
                    echo -ne "Error while uninstalling the Runtime Engine\n"
                    echo "[${NOW} install.sh] Error while uninstalling the RuntimeEngine." >> ./${LOGFILE}
                fi
                if [[ ${DEINSTERROR} == "0" ]]; then
                    rm -fR "/opt/GuH/" "/var/opt/GuH/" >> ./${LOGFILE} 2>&1
                    rpm -qa | grep -i '^sed-[0-9]' >> /dev/null 2>&1
                    if [[ ! $? == 0 ]]; then
                        if [[ ${REDHAT} == "TRUE" ]]; then
                            yum install -y sed >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                rpm -Uvh sed* >> ./${LOGFILE} 2>&1
                                if [[ ! $? == 0 ]]; then
                                    echo "[${NOW} install.sh] Unable to install the 'sed' package." >> ./${LOGFILE}
                                fi
                            fi
                        else
                            zypper --non-interactive in sed >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                rpm -Uvh sed* >> ./${LOGFILE} 2>&1
                                if [[ ! $? == 0 ]]; then
                                    echo "[${NOW} install.sh] Unable to install the 'sed' package." >> ./${LOGFILE}
                                fi
                            fi
                        fi
                    fi
                    sed -i -e '/.*DAC_RUNTIMEENGINE_URL.*/d' -e '/.*DAC_SYSTEM_URL.*/d' /etc/environment >> ./${LOGFILE} 2>&1
                    echo -e "\nThe daccord software components have been successfully removed."
                    echo "[${NOW} install.sh] The daccord software components have been successfully removed." >> ./${LOGFILE}
                else
                    echo -e "\033[1;31mError while uninstalling some of the components.\033[m"
                    echo -e "\033[1;31mTo troubleshoot the deinstallation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
                    CANCEL
                fi
            else
                echo "[${NOW} install.sh] User has chosen to skip the Runtime Engine." >> ./${LOGFILE}
                rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    echo -n "Please wait while the Runtime Engine is being started..."
                    echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
                    systemctl daemon-reload >> ./${LOGFILE} 2>&1
                    systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
                    if [[ $? == 0 ]]; then
                        sleep 15 >> /dev/null 2>&1
                        echo -ne "Runtime Engine started successfully\n"
                        echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
                    else
                        echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                        echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                        PRESS_ANY_KEY
                    fi
                fi
                rm -f /opt/GuH/daccord_ip_lifecycle_install_flag >> /dev/null
            fi
        else
            if [[ ${DEINSTERROR} == "0" ]]; then
                rm -fR "/opt/GuH/" "/var/opt/GuH/" >> ./${LOGFILE} 2>&1
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    else
        if [[ ${DEINSTERROR} == "0" ]]; then
            rm -f /opt/GuH/daccord_ip_lifecycle_install_flag >> /dev/null
            echo -e "\nThe daccord software components have been successfully removed."
            echo "[${NOW} install.sh] The daccord software components have been successfully removed." >> ./${LOGFILE}
            rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                echo -n "Please wait while the Runtime Engine is being started..."
                echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
                systemctl daemon-reload >> ./${LOGFILE} 2>&1
                systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
                if [[ $? == 0 ]]; then
                    sleep 15 >> /dev/null 2>&1
                    echo -ne "Runtime Engine started successfully\n"
                    echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
                else
                    echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                    echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                    PRESS_ANY_KEY
                fi
            fi
        else
            echo -e "\033[1;31mError while uninstalling some of the components.\033[m"
            echo -e "\033[1;31mTo troubleshoot the deinstallation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
            CANCEL
        fi
    fi
}

echo "[${NOW} install.sh] Script started." >> ./${LOGFILE} 2>&1
CHECK_ROOTPRIVILEGE
CHECK_OS
OUTPUT_WELCOME
CHECK_DEV
CHECK_GADGET_INST
CHECK_ENGINE_INST
CHECK_PREVINSTALL_FLAG
CHECK_SERVER_TYPE
while true; do
    if [[ ${PREVINSTALL} = 1 ]]; then
        EINGABE=
        while [[ ${EINGABE} != [rRcC] ]]; do
            echo -e "[${NOW} install.sh] A previous unfinished installation has been detected on this system." >> ./${LOGFILE}
            echo -e "\nA previous unfinished installation has been detected on this system."
            read -p $'\033[0;36mWould you like to [r] remove all installed components or [c] cancel this installation? [r/c] \033[m' EINGABE
            if [[ ${EINGABE} != [rRcC] ]]; then
                echo -e "\n\033[0;35mPlease choose between [r] to remove and [c] to cancel!\033[m"
            fi
        done
        if [[ ${EINGABE} == [rR] ]]; then
            UNINSTALL
            EXIT
        else
            CANCEL
        fi
    elif [[ ${DACCORD_GADGET_STAT} = 1 ]] || [[ ${DACCORD_ENGINE_STAT} = 1 ]]; then
        EINGABE=
        while [[ ${EINGABE} != [iIuUrRcC] ]]; do
            echo -e "\nThere is at least one daccord software component already installed on this system."
            read -p $'\033[0;36mWould you like to install [i] additional components, update [u] the current component(s), remove [r] the current component(s) and cancel [c] to exit the script? [i/u/r/c] \033[m' EINGABE
            if [[ ${EINGABE} != [iIuUrRcC] ]]; then
                echo -e "\n\033[0;35mPlease choose between [i] to install additional components, [u] to update the current component(s), [r] to remove the current component(s) and [c] to cancel and exit the script!\033[m"
            fi
        done
        if [[ ${EINGABE} == [iI] ]]; then
            EINGABE=
            while [[ ${EINGABE} != [oOlL] ]]; do
                echo -e "\nThe install script provides two installation modes."
                echo -e "     [o] Online mode - The required repositories will be added and the components will be downloaded from their respective repository."
                echo -e "     [l] Local mode - The components will be installed from local rpm files. These need to be present in the directory this script is executed from."
                read -p  $'\033[0;36mPlease enter the letter for the desired installation mode and confirm with the return key: [o/l] \033[m' EINGABE
                if [[ ${EINGABE} != [oOlL] ]]; then
                    echo -e "\n\033[0;35mPlease choose between [o] for online mode and [l] for local mode!\033[m"
                fi
            done
            if [[ ${EINGABE} == [oO] ]]; then
                SET_PREVINSTALL_FLAG
                REDNSORIP=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f3 | cut -d \: -f1)
                ONLINE_INSTALL
                PRESS_ANY_KEY
                exit 1
            else
                SET_PREVINSTALL_FLAG
                if ls ./*.rpm >> /dev/null 2>&1; then
                    REDNSORIP=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f3 | cut -d \: -f1)
                    OFFLINE_INSTALL
                    PRESS_ANY_KEY
                    exit 1
                else
                    echo -e "\nNo rpm files have been found. Please make sure you have all required rpm files in the same directory as the install script."
                    echo -e "[${NOW} install.sh] No rpm files have been found. Please make sure you have all required rpm files in the same directory as the install script." >> ./${LOGFILE}
                    CANCEL
                fi
            fi
        elif [[ ${EINGABE} == [uU] ]]; then
            EINGABE=
            while [[ ${EINGABE} != [pPcC] ]]; do
                echo -e "\033[0;31m\nAttention! Please ensure that a license with valid authorization for software updates is installed prior to performing the update. Otherwise, proper functionality of the software after the update cannot be guaranteed."
                read -p  $'\033[0;31mPlease confirm that a valid license is installed and decide if you wish to proceed [p] with the update or cancel [c] the update: [p/c] \033[m' EINGABE
                if [[ ${EINGABE} != [pPcC] ]]; then
                        echo -e "\nPlease choose between [p] to proceed with the update and [c] to cancel the update!"
                fi
            done
            if [[ ${EINGABE} == [cC] ]]; then
                EXIT
            else
                EINGABE=
                while [[ ${EINGABE} != [oOlL] ]]; do
                    echo -e "\nThe install script provides two update modes."
                    echo -e "     [o] Online mode - The components will be downloaded from their respective repositories and updated."
                    echo -e "     [l] Local mode - The components will be updated from local rpm files. These need to be present in the directory this script is executed from."
                    read -p  $'\033[0;36mPlease enter the letter for the desired update mode and confirm with the return key: [o/l] \033[m' EINGABE
                    if [[ ${EINGABE} != [oOlL] ]]; then
                        echo -e "\n\033[0;35mPlease choose between [o] for online mode and [l] for local mode!\033[m"
                    fi
                done
                if [[ ${EINGABE} == [oO] ]]; then
                    ONLINE_UPDATE
                else
                    OFFLINE_UPDATE
                fi
            fi
        elif [[ ${EINGABE} == [rR] ]]; then
            UNINSTALL
            EXIT
        else
            CANCEL
        fi
    else
        EINGABE=
        while [[ ${EINGABE} != [oOlL] ]]; do
            echo -e "\nThe install script provides two installation modes."
            echo -e "     [o] Online mode - The required repositories will be added and the components will be downloaded from their respective repository."
            echo -e "     [l] Local mode - The components will be installed from local rpm files. These need to be present in the directory this script is executed from."
            read -p  $'\033[0;36mPlease enter the letter for the desired installation mode and confirm with the return key: [o/l] \033[m' EINGABE
            if [[ ${EINGABE} != [oOlL] ]]; then
                echo -e "\n\033[0;35mPlease choose between [o] for online mode and [l] for local mode!\033[m"
            fi
        done
        if [[ ${EINGABE} == [oO] ]]; then
            SET_PREVINSTALL_FLAG
            grep "DAC_SYSTEM_URL=" /etc/environment >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                DAC_RUNTIMEENGINE_URL=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2)
                DAC_SYSTEM_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2)
                REDNSORIP=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f3 | cut -d \: -f1)
            else
                ENV_VARIABLES
            fi
            ONLINE_INSTALL
            PRESS_ANY_KEY
            exit 1
        else
            if ls ./*.rpm >> /dev/null 2>&1; then
                SET_PREVINSTALL_FLAG
                grep "DAC_SYSTEM_URL=" /etc/environment >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    DAC_RUNTIMEENGINE_URL=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2)
                    DAC_SYSTEM_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2)
                    REDNSORIP=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f3 | cut -d \: -f1)
                else
                    ENV_VARIABLES
                fi
                OFFLINE_INSTALL
                PRESS_ANY_KEY
                exit 1
            else
                echo -e "\nNo rpm files have been found. Please make sure you have all required rpm files in the same directory as the install script."
                echo -e "[${NOW} install.sh] No rpm files have been found. Please make sure you have all required rpm files in the same directory as the install script." >> ./${LOGFILE}
                CANCEL
            fi
        fi
    fi
done
