#!/bin/bash

##################################################################################################################
#                                                                                                                #
#                     daccord Integration Package Persons v.2.2.x - installation script                          #
#                                                                                                                #
##################################################################################################################

# Set stage to either "DEVELOPMENT" or "PRODUCTION" to influence what repository the rpm files are pulled from
STAGE="PRODUCTION"
# Timestamp variable used to timestamp log entries
NOW=$(date +"%d-%m-%Y %T")
# Date variable used to give the logfile a name including the current date + hour & minute of creation
DATE=$(date +"%Y-%m-%d-%H-%M-%S")
# Logfile variable that holds the name of the logifle including the $DATE variable from above
LOGFILE=daccord-ip-persons-install-${DATE}.log

# If the user id isn't 0, the script isn't being run as root and gives out a warning that it needs to be run as root to run successfully
CHECK_ROOTPRIVILEGE(){
    if [[ $EUID -ne 0 ]]; then
        echo -e "\033[0;31mThis script must be run with root privileges! Please verify that you have the needed privileges and restart the installation!\033[m"
        EXIT
    fi
}

CHECK_PRIMARY_SERVER(){
    if [[ ${SERVERTYPE} != "p" ]]; then
        echo -e "\033[0;31mThe daccord Integration Package Persons components can only be installed on a primary daccord server!\033[m"
        EXIT
    fi
}

# Check if the script is running on openSUSE or SLES or Redhat
CHECK_OS(){
    cat /etc/os-release >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        OS=$(cat /etc/os-release | grep -e "^NAME=")
        if [[ ${OS} == 'NAME="openSUSE Leap"' ]] || [[ ${OS} == 'NAME="SLES"' ]]; then
            REDHAT="FALSE"
        else
            REDHAT="TRUE"
        fi
    else
        REDHAT="TRUE"
    fi
}

# Check if the stage variable is set to "DEVELOPMENT" and warn about it if set as such
CHECK_DEV(){
    if [[ $STAGE == "DEVELOPMENT" ]]; then
        echo -e "\033[0;31m\nWARNING! - This script is running in DEVELOPMENT mode.\033[m"
        echo -e "\033[0;31mIt will install or update from DEVELOPMENT repositories.\033[m"
    fi
}

# Creates a file to let the script notice on the next run if the previous installation didn't run through entirely
SET_PREVINSTALL_FLAG(){
    mkdir -p /opt/GuH/ >> ./${LOGFILE} 2>&1
    touch /opt/GuH/daccord_ip_persons_install_flag >> ./${LOGFILE} 2>&1
}

# Check if the file that is being created by the above funtion exists on this system
CHECK_PREVINSTALL_FLAG(){
    if [[ -e /opt/GuH/daccord_ip_persons_install_flag ]]; then
        PREVINSTALL=1
    fi
}

CANCEL(){
    echo "Canceling the install script!"
    echo "[${NOW} install.sh] Script has been canceled." >> ./${LOGFILE}
    exit 1
}

EXIT(){
    exit 1
}

PRESS_ANY_KEY(){
    echo -e "Press any key to continue."
    read -sn 1
}

# Check if this system already has the daccord-persons-gadget installed
CHECK_GADGET_INST(){
    DACCORD_GADGET_STAT=-1
    if [[ -e /opt/GuH/daccord-persons-gadget ]]; then
        DACCORD_GADGET_STAT=1
    fi
}

CHECK_SERVER_TYPE(){
    SERVERTYPE=""
    if [[ -e /opt/GuH/daccord-system-gadget ]]; then
        SERVERTYPE="p"
    else
        SERVERTYPE="s"
    fi
}

OUTPUT_WELCOME(){
    clear
    echo -e "\033[0;36m################################################################################\033[m"
    echo -e "\033[0;36m#                                                                              #\033[m"
    echo -e "\033[0;36m#\033[m      \033[mdaccord Integration Package Persons v.2.2.x - Installation script\033[m       \033[0;36m#\033[m"
    echo -e "\033[0;36m#                                                                              #\033[m"
    echo -e "\033[0;36m################################################################################\033[m"
    echo -e " "
    echo -e "For detailed information please refer to the logfile: ${PWD}/${LOGFILE}"
}

ENV_VARIABLES(){
    REURLSCHEMA=
    while ! [[ "${REURLSCHEMA}" == [iIsS] ]]; do
        echo -e "\n\033[0;36mThe different daccord components communicate with each other. It is possibile to configure if the communication should be done securely (HTTPS and port 55443) or insecurely (HTTP and port 55080).\033[m"
        read -p $'\033[0;36mPlease choose either [i] for "Insecure" or [s] for "Secure" communication! [i/s] \033[m' REURLSCHEMA
        if [[ "${REURLSCHEMA}" == [iI] ]]; then
            export REURLSCHEMACHOSEN="http://"
            export REURLPORT=":55080"
        elif [[ "${REURLSCHEMA}" == [sS] ]]; then
            export REURLSCHEMACHOSEN="https://"
            export REURLPORT=":55443"
        fi
    done
    REDNSORIP=
    while ! [[ "${REDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z|-]*[0-9|a-zA-Z]$ ]]; do
        echo -e "\n\033[0;36mThis server needs to be accessible by the other components over the network.\033[m"
        read -p $'\033[0;36mPlease enter either the valid IP address or resolvable DNS name of this server: \033[m' REDNSORIP
        if [[ "${REDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z|-]*[0-9|a-zA-Z]$ ]]; then
            if [[ "${REDNSORIP}" =~ ^[0-9]+[.|0-9]*[0-9]$ ]]; then
                ip a | grep ${REDNSORIP} >> /dev/null 2>&1
                if [[ ! $? == 0 ]]; then
                    echo -e "\nThe IP Address that has been provided is not bound on any network adapter device on this server!"
                    REDNSORIP=
                else
                    export REDNSORIP
                fi
            else
                rpm -qa | grep -i '^bind-utils-[0-9]' >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    echo "[${NOW} install.sh] The 'bind-utils' package is already installed, proceeding with URL check." >> ./${LOGFILE}
                else
                    echo "[${NOW} install.sh] The 'bind-utils' package is not yet installed, proceeding with 'bind-utils' installation." >> ./${LOGFILE}
                    if [[ ${REDHAT} == "TRUE" ]]; then
                        yum install -y bind-utils >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh bind-utils*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'bind-utils' package." >> ./${LOGFILE}
                            fi
                        fi
                    else
                        zypper --non-interactive in bind-utils >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh bind-utils*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'bind-utils' package." >> ./${LOGFILE}
                            fi
                        fi
                    fi
                fi
                nslookup ${REDNSORIP} | grep -i "can't find" >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    echo -e "\nThe DNS Name that has been provided is not resolvable!"
                    REDNSORIP=
                else
                    export REDNSORIP
                fi
            fi
        fi
    done
    export DAC_RUNTIMEENGINE_URL="${REURLSCHEMACHOSEN}${REDNSORIP}${REURLPORT}/"
    if [[ ${SERVERTYPE} = [sS] ]]; then
        SYSURLSCHEMA=
        while ! [[ "${SYSURLSCHEMA}" == [iIsS] ]]; do
            echo -e "\n\033[0;36mThe different daccord components communicate with each other either securely (HTTPS and port 55443) or insecurely (HTTP and port 55080) depending on how the primary dacord server has been configured.\033[m"
            read -p $'\033[0;36mPlease choose either [i] for "Insecure" or [s] for "Secure" communication to the primary daccord server! [i/s] \033[m' SYSURLSCHEMA
            if [[ "${SYSURLSCHEMA}" == [iI] ]]; then
                export SYSURLSCHEMACHOSEN="http://"
                export SYSURLPORT=":55080"
            elif [[ "${SYSURLSCHEMA}" == [sS] ]]; then
                export SYSURLSCHEMACHOSEN="https://"
                export SYSURLPORT=":55443"
            fi
        done
        SYSDNSORIP=
        while ! [[ "${SYSDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z]*[0-9|a-zA-Z]$ ]]; do
            echo -e "\n\033[0;36mThis server needs to access the primary daccord server over the network.\033[m"
            read -p $'\033[0;36mPlease enter either the valid IP address or resolvable DNS name of the primary daccord server: \033[m' SYSDNSORIP
            if [[ "${SYSDNSORIP}" =~ ^[0-9|a-zA-Z]+[.|0-9|a-zA-Z]*[0-9|a-zA-Z]$ ]]; then
                export SYSDNSORIP
                export DAC_SYSTEM_URL="${SYSURLSCHEMACHOSEN}${SYSDNSORIP}${SYSURLPORT}/system/"
                rpm -qa | grep -i '^wget-[0-9]' >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    echo "[${NOW} install.sh] The 'wget' package is already installed, proceeding with URL check." >> ./${LOGFILE}
                else
                    echo "[${NOW} install.sh] The 'wget' package is not yet installed, proceeding with 'wget' installation." >> ./${LOGFILE}
                    if [[ ${REDHAT} == "TRUE" ]]; then
                        yum install -y wget >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh wget*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'wget' package." >> ./${LOGFILE}
                            fi
                        fi
                    else
                        zypper --non-interactive in wget >> ./${LOGFILE} 2>&1
                        if [[ ! $? == 0 ]]; then
                            rpm -Uvh wget*  >> ./${LOGFILE} 2>&1
                            if [[ ! $? == 0 ]]; then
                                echo "[${NOW} install.sh] Unable to install the 'wget' package." >> ./${LOGFILE}
                            fi
                        fi
                    fi
                    while true; do
                        rpm -qa | grep -i '^wget-[0-9]' >> /dev/null 2>&1
                        if [[ $? == 0 ]]; then
                            sleep 2
                            break
                        fi
                        echo -n '.'
                        sleep 1
                    done
                fi
                wget --no-check-certificate --timeout=5 --tries=1 ${DAC_SYSTEM_URL} >> ./${LOGFILE} 2>&1
                rm ./index.html >> /dev/null 2>&1
                if [[ ! $? == 0 ]]; then
                    echo -e "\nThe primary server was not reachable with the provided configuration. Please check and enter the configuration again!"
                    echo -e "Provided configuration: ${DAC_SYSTEM_URL}"
                    echo ""
                    DAC_SYSTEM_URL=""
                    SYSDNSORIP=""
                    SYSURLSCHEMA=""
                    while ! [[ "${SYSURLSCHEMA}" == [iIsS] ]]; do
                        echo -e "\n\033[0;36mThe different daccord components communicate with each other either securely (HTTPS and port 55443) or insecurely (HTTP and port 55080) depending on how the primary dacord server has been configured.\033[m"
                        read -p $'\033[0;36mPlease choose either [i] for "Insecure" or [s] for "Secure" communication to the primary daccord server! [i/s] \033[m' SYSURLSCHEMA
                        if [[ "${SYSURLSCHEMA}" == [iI] ]]; then
                            export SYSURLSCHEMACHOSEN="http://"
                            export SYSURLPORT=":55080"
                        elif [[ "${SYSURLSCHEMA}" == [sS] ]]; then
                            export SYSURLSCHEMACHOSEN="https://"
                            export SYSURLPORT=":55443"
                        fi
                    done
                fi
            fi
        done
        export DAC_SYSTEM_URL="${SYSURLSCHEMACHOSEN}${SYSDNSORIP}${SYSURLPORT}/system/"
    else
        export DAC_SYSTEM_URL="${DAC_RUNTIMEENGINE_URL}system/"
    fi
    echo "DAC_RUNTIMEENGINE_URL='${DAC_RUNTIMEENGINE_URL}'" >> /etc/environment
    echo "DAC_SYSTEM_URL='${DAC_SYSTEM_URL}'" >> /etc/environment
    source /etc/environment
}

ONLINE_INSTALL(){
    echo "[${NOW} install.sh] User has requested to start the online installation." >> ./${LOGFILE}
    INSTERROR=0
    if [[ ${REDHAT} == "TRUE" ]]; then
        ONLINE_REPO_INSTALL_REDHAT
    else
        ONLINE_REPO_INSTALL_SUSE
    fi
    if [[ $INSTERROR == "0" ]]; then
        echo -e "\n\033[0;36mThe daccord software components have been installed successfully!\033[m"
        if [[ ! ${DACCORD_GADGET_STAT} = 1 ]]; then
            DACCORD_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f1,2,3)
            echo -e "\033[0;36mTo finalize your installation, please access the URL ${DACCORD_URL}/daccord/ with a web-browser to configure your daccord system.\033[m"
        fi
        echo -e "\n"
        rm -f /opt/GuH/daccord_ip_persons_install_flag >> /dev/null
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the installation of the daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the installation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
    fi
}

ONLINE_REPO_INSTALL_SUSE(){
    echo "[${NOW} install.sh] Installing the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper lr daccord-ip-persons-2.2-dev >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-persons-2.2-dev daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        zypper lr daccord-ip-persons-2.2 >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-persons-2.2 daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    if [[ ${SERVERTYPE} == [pP] ]]; then
        rpm -qa | grep -i '^daccord-persons-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Persons Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Persons Gadget." >> ./${LOGFILE}
            zypper --non-interactive in daccord-persons-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Persons Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-persons-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Persons Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-persons-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    systemctl status RuntimeEngine.service |grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being started..."
        echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
        systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "The Runtime Engine has been successfully started\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper mr -d daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
    else
        zypper mr -d daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
    fi
}

ONLINE_REPO_INSTALL_REDHAT(){
    echo "[${NOW} install.sh] Installing the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        if [[ ! -f "/etc/yum.repos.d/daccord-ip-persons-2.2-dev.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-ip-persons-2.2-dev.repo
[daccord-ip-persons-2.2-dev]
name=daccord IP Persons Development RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-persons-2.2-dev
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        if [[ ! -f "/etc/yum.repos.d/daccord-ip-persons-2.2.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-ip-persons-2.2.repo
[daccord-ip-persons-2.2]
name=daccord IP Persons RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-persons-2.2
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    if [[ ${SERVERTYPE} == [pP] ]]; then
        rpm -qa | grep -i '^daccord-persons-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Persons Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Persons Gadget." >> ./${LOGFILE}
            yum install -y daccord-persons-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Persons Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-persons-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Persons Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-persons-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    systemctl status RuntimeEngine.service |grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being started..."
        echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
        systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "The Runtime Engine has been successfully started\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        yum config-manager --disable daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
    else
        yum config-manager --disable daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
    fi
}

OFFLINE_INSTALL(){
    echo "[${NOW} install.sh] User has requested to start the offline installation." >> ./${LOGFILE}
    INSTERROR=0
    OFFLINE_RPM_INSTALL
    if [[ $INSTERROR == "0" ]]; then
        echo -e "\nThe daccord software components have been installed successfully!"
        if [[ ! ${DACCORD_CURRENT_STAT} = 1 ]]; then
            DACCORD_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f1,2,3)
            echo -e "\033[0;36mTo finalize your installation, please access the URL ${DACCORD_URL}/daccord/ with a web-browser to configure your daccord system.\033[m"
        fi
        echo -e "\n"
        rm -f /opt/GuH/daccord_ip_persons_install_flag >> /dev/null
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the installation of the daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the installation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
    fi
}

OFFLINE_RPM_INSTALL(){
    echo "[${NOW} install.sh] Installing the daccord software components from local rpm files" >> ./${LOGFILE}
    if [[ ${SERVERTYPE} == [pP] ]]; then
        rpm -qa | grep -i '^daccord-persons-gadget-[0-9]' >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Persons Gadget is being installed..."
            echo "[${NOW} install.sh] User has chosen to install the Persons Gadget." >> ./${LOGFILE}
            rpm -Uvh daccord-persons-gadget* >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Persons Gadget installed successfully\n"
                echo "[${NOW} install.sh] daccord-persons-gadget installed successfully." >> ./${LOGFILE}
            else
                INSTERROR=1
                echo -ne "Error while installing the Persons Gadget\n"
                echo "[${NOW} install.sh] Error while installing the daccord-persons-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    systemctl status RuntimeEngine.service |grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being started..."
        echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
        systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "The Runtime Engine has been successfully started\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
}

ONLINE_UPDATE(){
    echo "[${NOW} install.sh] User has requested to start the online update." >> ./${LOGFILE}
    UPERROR=0
    echo ""
    if [[ ${REDHAT} == "TRUE" ]]; then
        ONLINE_REPO_UPDATE_REDHAT
    else
        ONLINE_REPO_UPDATE_SUSE
    fi
    if [[ $UPERROR == "0" ]]; then
        echo -e "\nThe daccord software components have been updated successfully!"
        DACCORD_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f1,2,3)
        echo -e "\033[0;36mTo finalize your installation, please access the URL ${DACCORD_URL}/daccord/ with a web-browser to configure your daccord system.\033[m"
        echo -e "\n"
        EXIT
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the update of the daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the update, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
        EXIT
    fi
}

ONLINE_REPO_UPDATE_SUSE(){
    echo "[${NOW} install.sh] Updating the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper lr daccord-ip-persons-2.2-dev >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh --force daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-persons-2.2-dev daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        zypper lr daccord-ip-persons-2.2 >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper mr -e daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh --force daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
        else
            zypper ar http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-persons-2.2 daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
            zypper --gpg-auto-import-keys refresh daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -ne "ERROR! No Runtime Engine found on the system! Cancelling the update!"
        echo "[${NOW} install.sh] ERROR! No RuntimeEngine found on the system! Cancelling the update!" >> ./${LOGFILE}
        CANCEL
    fi
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        REMANUALSTOP="TRUE"
        echo "[${NOW} install.sh] The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!" >> ./${LOGFILE}
        echo -e "The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!\n"
    else
        REMANUALSTOP="FALSE"
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for additional installations..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    rpm -qa | grep -i '^daccord-persons-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        zypper --non-interactive up -D daccord-persons-gadget | grep "highest available version" >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -e "The are currently no updates for the Persons Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-persons-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Persons Gadget is being updated..."
            zypper --non-interactive up daccord-persons-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Persons Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-persons-gadget updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Persons Gadget\n"
                echo "[${NOW} install.sh] Error while updating the daccord-persons-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    if [[ ${REMANUALSTOP} == "FALSE" ]]; then
        systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Runtime Engine is being started..."
            echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
            systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 15 >> /dev/null 2>&1
                echo -ne "The Runtime Engine has been successfully started\n"
                echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        zypper mr -d daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
    else
        zypper mr -d daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
    fi
}

ONLINE_REPO_UPDATE_REDHAT(){
    echo "[${NOW} install.sh] Updating the daccord software components from the repository." >> ./${LOGFILE}
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        if [[ ! -f "/etc/yum.repos.d/daccord-ip-persons-2.2-dev.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-ip-persons-2.2-dev.repo
[daccord-ip-persons-2.2-dev]
name=daccord IP Persons Development RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-persons-2.2-dev
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
        fi
    else
        if [[ ! -f "/etc/yum.repos.d/daccord-ip-persons-2.2.repo" ]]; then
cat <<EOF>  /etc/yum.repos.d/daccord-ip-persons-2.2.repo
[daccord-ip-persons-2.2]
name=daccord IP Persons RPM Repository
baseurl=http://mirror.guh-systems.de/products/GuH/repos/update/daccord-ip-persons-2.2
enabled=1
gpgcheck=0
EOF
        else
            yum config-manager --enable daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    yum clean all >> /dev/null 2>&1
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -ne "ERROR! No Runtime Engine found on the system! Cancelling the update!"
        echo "[${NOW} install.sh] ERROR! No RuntimeEngine found on the system! Cancelling the update!" >> ./${LOGFILE}
        CANCEL
    fi
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        REMANUALSTOP="TRUE"
        echo "[${NOW} install.sh] The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!" >> ./${LOGFILE}
        echo -e "The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!\n"
    else
        REMANUALSTOP="FALSE"
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for additional installations..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    rpm -qa | grep -i '^daccord-persons-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        yum check-update daccord-persons-gadget | grep daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
        if [[ ! $? == 0 ]]; then
            echo -e "The are currently no updates for the Persons Gadget\n"
            echo "[${NOW} install.sh] The are currently no updates for the daccord-persons-gadget." >> ./${LOGFILE}
        else
            echo -n "Please wait while the Persons Gadget is being updated..."
            yum upgrade -y daccord-persons-gadget >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                echo -ne "Persons Gadget updated successfully\n"
                echo "[${NOW} install.sh] daccord-persons-gadget updated successfully." >> ./${LOGFILE}
            else
                UPERROR=1
                echo -ne "Error while updating the Persons Gadget\n"
                echo "[${NOW} install.sh] Error while updating the daccord-persons-gadget." >> ./${LOGFILE}
            fi
        fi
    fi
    systemctl daemon-reload >> ./${LOGFILE} 2>&1
    if [[ ${REMANUALSTOP} == "FALSE" ]]; then
        systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
        if [[ ! $? == 0 ]]; then
            echo -n "Please wait while the Runtime Engine is being started..."
            echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
            systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 15 >> /dev/null 2>&1
                echo -ne "The Runtime Engine has been successfully started\n"
                echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
    if [[ ${STAGE} == "DEVELOPMENT" ]]; then
        yum config-manager --disable daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
    else
        yum config-manager --disable daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
    fi
}

OFFLINE_UPDATE(){
    echo "[${NOW} install.sh] User has requested to start the offline update." >> ./${LOGFILE}
    UPERROR=0
    echo ""
    OFFLINE_RPM_UPDATE
    if [[ $UPERROR == "0" ]]; then
        echo -e "\nThe daccord software components have been updated successfully!"
        DACCORD_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f1,2,3)
        echo -e "\033[0;36mTo finalize your installation, please access the URL ${DACCORD_URL}/daccord/ with a web-browser to configure your daccord system.\033[m"
        echo -e "\n"
        EXIT
    else
        echo -e "\n"
        echo -e "\033[1;31mSomething went wrong! An error was detected during the update of the daccord software components!\033[m"
        echo -e "\033[1;31mTo troubleshoot the update, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        echo -e "\n"
        EXIT
    fi
}

OFFLINE_RPM_UPDATE(){
    echo "[${NOW} install.sh] Updating the daccord software components from local rpm files." >> ./${LOGFILE}
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        echo -ne "ERROR! No Runtime Engine found on the system! Cancelling the update!"
        echo "[${NOW} install.sh] ERROR! No RuntimeEngine found on the system! Cancelling the update!" >> ./${LOGFILE}
        CANCEL
    fi
    systemctl status RuntimeEngine.service | grep -i "active (running)" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        REMANUALSTOP="TRUE"
        echo "[${NOW} install.sh] The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!" >> ./${LOGFILE}
        echo -e "The RuntimeEngine has been stopped manually outside the update process. The RuntimeEngine will NOT be started after the update process has finished!\n"
    else
        REMANUALSTOP="FALSE"
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for component updates..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    rpm -qa | grep -i '^daccord-persons-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Persons Gadget is being updated..."
        rpm -Uvh daccord-persons-gadget* >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Persons Gadget updated successfully\n"
            echo "[${NOW} install.sh] daccord-persons-gadget updated successfully." >> ./${LOGFILE}
        else
            UPERROR=1
            echo -ne "Error while updating the Persons Gadget\n"
            echo "[${NOW} install.sh] Error while updating the daccord-persons-gadget." >> ./${LOGFILE}
        fi
    fi
    systemctl status RuntimeEngine.service |grep -i "RuntimeEngine.sh; enabled;" >> /dev/null 2>&1
    if [[ ! $? == 0 ]]; then
        systemctl enable RuntimeEngine.service >> ./${LOGFILE} 2>&1
    fi
}

UNINSTALL(){
    DEINSTERROR=0
    echo "[${NOW} install.sh] User has requested to start the deinstallation." >> ./${LOGFILE}
    if [[ ${REDHAT} == "TRUE" ]]; then
        if [[ -f "/etc/yum.repos.d/daccord-ip-persons-2.2-dev.repo" ]]; then
            rm -f /etc/yum.repos.d/daccord-ip-persons-2.2-dev.repo >> ./${LOGFILE} 2>&1
        fi
        if [[ -f "/etc/yum.repos.d/daccord-ip-persons-2.2.repo" ]]; then
            rm -f /etc/yum.repos.d/daccord-ip-persons-2.2.repo >> ./${LOGFILE} 2>&1
        fi
    else
        zypper lr daccord-ip-persons-2.2-dev >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper rr daccord-ip-persons-2.2-dev >> ./${LOGFILE} 2>&1
        fi
        zypper lr daccord-ip-persons-2.2 >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            zypper rr daccord-ip-persons-2.2 >> ./${LOGFILE} 2>&1
        fi
    fi
    rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Runtime Engine is being stopped to prepare for component deinstallations..."
        echo "[${NOW} install.sh] Trying to stop the RuntimeEngine." >> ./${LOGFILE}
        systemctl stop RuntimeEngine.service >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            sleep 15 >> /dev/null 2>&1
            echo -ne "Runtime Engine stopped successfully\n"
            echo "[${NOW} install.sh] The RuntimeEngine has been successfully stopped." >> ./${LOGFILE}
        else
            echo -ne "Error while trying to stop the Runtime Engine. Please open a new console and try to stop the Runtime Engine manually with 'systemctl stop RuntimeEngine.service' before proceeding!\n"
            echo "[${NOW} install.sh] Error while trying to stop the RuntimeEngine." >> ./${LOGFILE}
            PRESS_ANY_KEY
        fi
    fi
    rpm -qa | grep -i '^daccord-persons-gadget-[0-9]' >> /dev/null 2>&1
    if [[ $? == 0 ]]; then
        echo -n "Please wait while the Persons Gadget is being uninstalled..."
        echo "[${NOW} install.sh] User has chosen to uninstall the Persons Gadget." >> ./${LOGFILE}
        rpm -ve daccord-persons-gadget >> ./${LOGFILE} 2>&1
        if [[ $? == 0 ]]; then
            echo -ne "Persons Gadget uninstalled successfully\n"
            echo "[${NOW} install.sh] daccord-persons-gadget uninstalled successfully." >> ./${LOGFILE}
            rm -fR "/opt/GuH/daccord-persons-gadget" "/var/opt/GuH/daccord-persons-gadget" "/opt/GuH/RuntimeEngine/webapps/persons" "/opt/GuH/RuntimeEngine/webapps/persons.war" >> ./${LOGFILE} 2>&1
        else
            DEINSTERROR=1
            echo -ne "Error while uninstalling the Persons Gadget\n"
            echo "[${NOW} install.sh] Error while uninstalling the daccord-persons-gadget." >> ./${LOGFILE}
        fi
    fi
    if [[ ${DEINSTERROR} == "0" ]]; then
        rm -f /opt/GuH/daccord_ip_persons_install_flag >> /dev/null
        echo -e "\nThe daccord software components have been successfully removed."
        echo "[${NOW} install.sh] The daccord software components have been successfully removed." >> ./${LOGFILE}
        rpm -qa | grep -i '^RuntimeEngine-[0-9]' >> /dev/null 2>&1
        if [[ $? == 0 ]]; then
            echo -n "Please wait while the Runtime Engine is being started..."
            echo "[${NOW} install.sh] Trying to start the RuntimeEngine." >> ./${LOGFILE}
            systemctl daemon-reload >> ./${LOGFILE} 2>&1
            systemctl start RuntimeEngine.service >> ./${LOGFILE} 2>&1
            if [[ $? == 0 ]]; then
                sleep 15 >> /dev/null 2>&1
                echo -ne "Runtime Engine started successfully\n"
                echo "[${NOW} install.sh] The RuntimeEngine has been successfully started." >> ./${LOGFILE}
            else
                echo -ne "Error while trying to start the Runtime Engine. Please open a new console and try to start the Runtime Engine manually with 'systemctl start RuntimeEngine.service' before proceeding!\n"
                echo "[${NOW} install.sh] Error while trying to start the RuntimeEngine." >> ./${LOGFILE}
                PRESS_ANY_KEY
            fi
        fi
    else
        echo -e "\033[1;31mError while uninstalling some of the components.\033[m"
        echo -e "\033[1;31mTo troubleshoot the deinstallation, please refer to logfile: ${PWD}/${LOGFILE} \033[m"
        CANCEL
    fi
}

echo "[${NOW} install.sh] Script started." >> ./${LOGFILE} 2>&1
CHECK_ROOTPRIVILEGE
CHECK_OS
OUTPUT_WELCOME
CHECK_DEV
CHECK_GADGET_INST
CHECK_PREVINSTALL_FLAG
CHECK_SERVER_TYPE
while true; do
    if [[ ${PREVINSTALL} = 1 ]]; then
        EINGABE=
        while [[ ${EINGABE} != [rRcC] ]]; do
            echo -e "[${NOW} install.sh] A previous unfinished installation has been detected on this system." >> ./${LOGFILE}
            echo -e "\nA previous unfinished installation has been detected on this system."
            read -p $'\033[0;36mWould you like to [r] remove all installed components or [c] cancel this installation? [r/c] \033[m' EINGABE
            if [[ ${EINGABE} != [rRcC] ]]; then
                echo -e "\n\033[0;35mPlease choose between [r] to remove and [c] to cancel!\033[m"
            fi
        done
        if [[ ${EINGABE} == [rR] ]]; then
            UNINSTALL
            EXIT
        else
            CANCEL
        fi
    elif [[ ${DACCORD_GADGET_STAT} = 1 ]]; then
        EINGABE=
        while [[ ${EINGABE} != [uUrRcC] ]]; do
            echo -e "\nThere is at least one daccord software component already installed on this system."
            read -p $'\033[0;36mWould you like to update [u] the components, remove [r] the components or cancel [c] to exit the script? [u/r/c] \033[m' EINGABE
            if [[ ${EINGABE} != [uUrRcC] ]]; then
                echo -e "\n\033[0;35mPlease choose between update [u] the components, remove [r] the components or cancel [c] to exit the script!\033[m"
            fi
        done
        if [[ ${EINGABE} == [uU] ]]; then
            EINGABE=
            while [[ ${EINGABE} != [pPcC] ]]; do
                echo -e "\033[0;31m\nAttention! Please ensure that a license with valid authorization for software updates is installed prior to performing the update. Otherwise, proper functionality of the software after the update cannot be guaranteed."
                read -p  $'\033[0;31mPlease confirm that a valid license is installed and decide if you wish to proceed [p] with the update or cancel [c] the update: [p/c] \033[m' EINGABE
                if [[ ${EINGABE} != [pPcC] ]]; then
                        echo -e "\nPlease choose between [p] to proceed with the update and [c] to cancel the update!"
                fi
            done
            if [[ ${EINGABE} == [cC] ]]; then
                EXIT
            else
                EINGABE=
                while [[ ${EINGABE} != [oOlL] ]]; do
                    echo -e "\nThe install script provides two update modes."
                    echo -e "     [o] Online mode - The components will be downloaded from their respective repositories and updated."
                    echo -e "     [l] Local mode - The components will be updated from local rpm files. These need to be present in the directory this script is executed from."
                    read -p  $'\033[0;36mPlease enter the letter for the desired update mode and confirm with the return key: [o/l] \033[m' EINGABE
                    if [[ ${EINGABE} != [oOlL] ]]; then
                        echo -e "\n\033[0;35mPlease choose between [o] for online mode and [l] for local mode!\033[m"
                    fi
                done
                if [[ ${EINGABE} == [oO] ]]; then
                    ONLINE_UPDATE
                else
                    OFFLINE_UPDATE
                fi
            fi
        elif [[ ${EINGABE} == [rR] ]]; then
            UNINSTALL
            EXIT
        else
            CANCEL
        fi
    else
        EINGABE=
        while [[ ${EINGABE} != [oOlL] ]]; do
            echo -e "\nThe install script provides two installation modes."
            echo -e "     [o] Online mode - The required repositories will be added and the components will be downloaded from their respective repository."
            echo -e "     [l] Local mode - The components will be installed from local rpm files. These need to be present in the directory this script is executed from."
            read -p  $'\033[0;36mPlease enter the letter for the desired installation mode and confirm with the return key: [o/l] \033[m' EINGABE
            if [[ ${EINGABE} != [oOlL] ]]; then
                echo -e "\n\033[0;35mPlease choose between [o] for online mode and [l] for local mode!\033[m"
            fi
        done
        if [[ ${EINGABE} == [oO] ]]; then
            SET_PREVINSTALL_FLAG
            grep "DAC_SYSTEM_URL=" /etc/environment >> /dev/null 2>&1
            if [[ $? == 0 ]]; then
                DAC_RUNTIMEENGINE_URL=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2)
                DAC_SYSTEM_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2)
                REDNSORIP=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f3 | cut -d \: -f1)
            else
                ENV_VARIABLES
            fi
            ONLINE_INSTALL
            PRESS_ANY_KEY
            exit 1
        else
            if ls ./*.rpm >> /dev/null 2>&1; then
                SET_PREVINSTALL_FLAG
                grep "DAC_SYSTEM_URL=" /etc/environment >> /dev/null 2>&1
                if [[ $? == 0 ]]; then
                    DAC_RUNTIMEENGINE_URL=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2)
                    DAC_SYSTEM_URL=$(grep "DAC_SYSTEM_URL=" /etc/environment | cut -d \' -f2)
                    REDNSORIP=$(grep "DAC_RUNTIMEENGINE_URL=" /etc/environment | cut -d \' -f2 | cut -d \/ -f3 | cut -d \: -f1)
                else
                    ENV_VARIABLES
                fi
                OFFLINE_INSTALL
                PRESS_ANY_KEY
                exit 1
            else
                echo -e "\nNo rpm files have been found. Please make sure you have all required rpm files in the same directory as the install script."
                echo -e "[${NOW} install.sh] No rpm files have been found. Please make sure you have all required rpm files in the same directory as the install script." >> ./${LOGFILE}
                CANCEL
            fi
        fi
    fi
done
